/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.event;

import java.io.Serializable;
import java.util.Map;

import javax.activation.DataHandler;

import org.mule.runtime.api.message.Attributes;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.internal.message.InternalMessage;

public class MessageBuilderDecorator {

  private InternalMessage.Builder builder;

  public MessageBuilderDecorator() {
    builder = (InternalMessage.Builder) InternalMessage.builder();
  }

  public MessageBuilderDecorator(Message originalMessage) {
    this.builder = InternalMessage.builder(originalMessage);
  }

  public MessageBuilderDecorator withInboundProperty(String key, Serializable value) {
    MediaType mediaType = DataType.fromObject(value).getMediaType();
    String mediaTypeStr = mediaType.getPrimaryType() + "/" + mediaType.getSubType();
    return withInboundProperty(key, value, mediaTypeStr, null);
  }

  public MessageBuilderDecorator withInboundProperty(String key, Object value, String mediaType, String charset) {
    DataType dataType = new DataTypeBuilder().value(value).encoding(charset).mediaType(mediaType).build();
    builder.addInboundProperty(key, new TypedValue(value, dataType));
    return this;
  }

  public MessageBuilderDecorator withInboundProperties(Map<String, TypedValue<Serializable>> properties) {
    if (null != properties && !properties.isEmpty()) {
      properties.forEach((k, v) -> builder.addInboundProperty(k, v.getValue(), v.getDataType()));
    }
    return this;
  }

  public MessageBuilderDecorator withOutboundProperty(String key, Serializable value) {
    MediaType mediaType = DataType.fromObject(value).getMediaType();
    String mediaTypeStr = mediaType.getPrimaryType() + "/" + mediaType.getSubType();
    return withOutboundProperty(key, value, mediaTypeStr, null);
  }

  public MessageBuilderDecorator withOutboundProperty(String key, Object value, String mediaType, String charset) {
    DataType dataType = new DataTypeBuilder().value(value).encoding(charset).mediaType(mediaType).build();
    builder.addOutboundProperty(key, new TypedValue(value, dataType));
    return this;
  }

  public MessageBuilderDecorator withOutboundProperties(Map<String, TypedValue<Serializable>> properties) {
    if (null != properties && !properties.isEmpty()) {
      properties.forEach((k, v) -> builder.addOutboundProperty(k, v.getValue(), v.getDataType()));
    }
    return this;
  }

  public MessageBuilderDecorator withInboundAttachment(String key, DataHandler value) {
    builder.addInboundAttachment(key, value);
    return this;
  }

  public MessageBuilderDecorator withInboundAttachments(Map<String, DataHandler> attachments) {
    if (null != attachments && !attachments.isEmpty()) {
      attachments.forEach((k, v) -> builder.addInboundAttachment(k, v));
    }
    return this;
  }

  public MessageBuilderDecorator withOutboundAttachment(String key, DataHandler value) {
    builder.addOutboundAttachment(key, value);
    return this;
  }

  public MessageBuilderDecorator withOutboundAttachments(Map<String, DataHandler> attachments) {
    if (null != attachments && !attachments.isEmpty()) {
      attachments.forEach((k, v) -> builder.addOutboundAttachment(k, v));
    }
    return this;
  }

  public MessageBuilderDecorator withPayload(Object payload) {
    builder.payload(payload);
    return this;
  }

  public MessageBuilderDecorator withMediaType(MediaType mediaType) {
    builder.mediaType(mediaType);
    return this;
  }

  public MessageBuilderDecorator withAttributes(Attributes attributes) {
    builder.attributes(attributes);
    return this;
  }

  public Message build() {
    return builder.build();
  }
}
