/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.properties;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.common.util.ClassUtils;
import org.mule.munit.common.util.IOUtils;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;

/**
 * It knows how to load and read the file mule-deploy.properties from the classpath. As long as the file belongs to the current
 * app and/or the related domain.
 */

public class MuleDeployPropertyLoader {

  public static final String DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE = "mule-deploy.properties";

  private final File appDir;

  private transient Log log = LogFactory.getLog(this.getClass());
  private Map<String, String> domainDeployPropertiesMap = null;
  private Map<String, String> applicationDeployPropertiesMap = null;

  public MuleDeployPropertyLoader(File appDir) {
    this.appDir = appDir;
  }

  public Map<String, String> getApplicationDeployProperties() {
    if (null == applicationDeployPropertiesMap) {
      loadApplicationDeployProperties();
    }
    return applicationDeployPropertiesMap;
  }

  public Map<String, String> getDomainDeployProperties() {
    if (null == applicationDeployPropertiesMap) {
      loadApplicationDeployProperties();
    }

    if (null == domainDeployPropertiesMap) {
      String domainName = applicationDeployPropertiesMap.get("domain");
      loadDomainDeployProperties(domainName);
    }

    return domainDeployPropertiesMap;
  }

  private void loadApplicationDeployProperties() {
    applicationDeployPropertiesMap = new HashMap<>();
    File deployPropertiesFile = new File(appDir, DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE);
    if (deployPropertiesFile.exists()) {
      log.debug("Loaded " + DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " file from:" + deployPropertiesFile);
      loadDeployProperties(deployPropertiesFile, applicationDeployPropertiesMap);
    }
  }

  private void loadDomainDeployProperties(String domainName) {
    domainDeployPropertiesMap = new HashMap<>();

    if (StringUtils.isBlank(domainName)) {
      log.debug("Attempting to load " + DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE
          + " for a blank domain will return not properties");
    }
    Enumeration<URL> urls = ClassUtils.getResources(DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE, getClass().getClassLoader());
    Optional<URL> url = Collections.list(urls).stream().filter(u -> doesTheFileBelongToTheDomain(u, domainName)).findFirst();

    if (url.isPresent()) {
      log.debug("Loaded " + DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " file from:" + url.get().getPath());
      loadDeployProperties(url.get(), domainDeployPropertiesMap);
    }
  }

  private void loadDeployProperties(File resourceFile, Map<String, String> propertiesMap) {
    try {
      loadDeployProperties(resourceFile.toURI().toURL(), propertiesMap);
    } catch (MalformedURLException e) {
      log.warn(DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " could not be loaded.");
    }
  }

  private void loadDeployProperties(URL resourceURL, Map<String, String> propertiesMap) {
    if (null == resourceURL) {
      log.warn(DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " file was not found");
      return;
    }
    try {
      propertiesMap.putAll(loadProperties(resourceURL));
    } catch (IOException e) {
      log.warn(DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " could not be loaded.");
    }
  }

  private boolean doesTheFileBelongToTheDomain(URL url, String domainName) {
    String domainMuleDeployPropertiesRelativePath = File.separator + domainName + File.separator;
    String path = new File(url.getPath()).getPath();

    if (path.contains(domainMuleDeployPropertiesRelativePath)) {
      return true;
    }
    return false;
  }

  private Map<String, String> loadProperties(URL propsURL) throws IOException {
    Map<String, String> appPropsMap = new HashMap<>();
    Properties props = IOUtils.toProperties(propsURL);
    for (Object key : props.keySet()) {
      appPropsMap.put(key.toString(), props.getProperty(key.toString()));
    }
    return appPropsMap;
  }

}
