package org.mule.munit.common.protocol.message;

import com.google.gson.Gson;
import org.mule.munit.common.protocol.listeners.RunEventListener;

import static org.mule.munit.common.protocol.message.MessageField.*;
import static org.mule.munit.common.protocol.message.MessageID.*;

/**
 * The goal of this class is to parse a string representing a {@link RunMessage} and react to it by calling the different methods
 * of the interface {@link RunEventListener}
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunMessageParser {

  private RunEventListener listener;

  public RunMessageParser(RunEventListener listener) {
    this.listener = listener;
  }

  public void parseAndNotify(String message) {
    RunMessage cmd = new Gson().fromJson(message, RunMessage.class);
    switch (cmd.getId()) {
      case RUN_START:
        listener.notifyRunStart();
        break;
      case SUITE_START:
        listener.notifySuiteStart(cmd.get(MUNIT_SUITE_KEY), Integer.valueOf(cmd.get(NUMBER_OF_TESTS_KEY)));
        break;
      case CONTAINER_START_FAILURE:
        listener.notifyContainerFailure(cmd.get(MUNIT_SUITE_KEY), cmd.get(STACK_TRACE_KEY));
        break;
      case BEFORE_SUITE_START:
        listener.notifyBeforeSuiteStart(cmd.get(NAME_KEY));
        break;
      case BEFORE_SUITE_END:
        listener.notifyBeforeSuiteEnd(cmd.get(NAME_KEY), cmd.get(STACK_TRACE_KEY), cmd.getStatus(STATUS_KEY));
        break;
      case TEST_START:
        listener.notifyTestStart(cmd.get(NAME_KEY));
        break;
      case TEST_END:
        listener.notifyTestEnd(cmd.get(NAME_KEY), cmd.get(STACK_TRACE_KEY), cmd.getStatus(STATUS_KEY),
                               cmd.getLong(ELAPSED_TIME_KEY));
        break;
      case AFTER_SUITE_START:
        listener.notifyAfterSuiteStart(cmd.get(NAME_KEY));
        break;
      case AFTER_SUITE_END:
        listener.notifyAfterSuiteEnd(cmd.get(NAME_KEY), cmd.get(STACK_TRACE_KEY), cmd.getStatus(STATUS_KEY));
        break;
      case SUITE_END:
        listener.notifySuiteEnd(cmd.get(MUNIT_SUITE_KEY), cmd.getLong(ELAPSED_TIME_KEY));
        break;
      case RUN_FINISH:
        listener.notifyRunFinish();
        break;
      case COVERAGE_REPORT:
        listener.notifyCoverageReport(cmd.get(COVERAGE_REPORT_KEY));
        break;
      case UNEXPECTED_SUITE_ERROR:
        listener.notifySuiteUnexpectedError(cmd.get(STACK_TRACE_KEY));
        break;
      case UNEXPECTED_ERROR:
        listener.notifyUnexpectedError(cmd.get(STACK_TRACE_KEY));
        break;
      default:
        listener.notifyUnknown(message);
        break;

    }
  }

}
