/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.protocol.notifiers;

import com.google.gson.Gson;
import org.apache.commons.lang3.StringUtils;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.message.RunMessage;
import org.mule.munit.common.protocol.message.TestStatus;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.ImmutableMap.of;
import static org.mule.munit.common.protocol.message.MessageField.*;
import static org.mule.munit.common.protocol.message.MessageID.*;

/**
 * <p>Abstract RunMessageNotifier class.</p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public abstract class RunMessageNotifier implements RunEventListener {

  protected String runToken;
  private Gson gson = new Gson();

  public abstract void init() throws Exception;

  public void setRunToken(String runToken) {
    this.runToken = runToken;
  }

  @Override
  public void notifyRunStart() {
    sendMessage(RUN_START, Collections.emptyMap());
  }

  @Override
  public void notifySuiteStart(String path, int numberOfTests) {
    sendMessage(SUITE_START, of(MUNIT_SUITE_KEY, path, NUMBER_OF_TESTS_KEY, String.valueOf(numberOfTests)));
  }

  @Override
  public void notifyContainerFailure(String suite, String stackTrace) {
    sendMessage(CONTAINER_START_FAILURE, of(MUNIT_SUITE_KEY, suite, STACK_TRACE_KEY, stackTrace));
  }

  @Override
  public void notifyBeforeSuiteStart(String name) {
    sendMessage(BEFORE_SUITE_START, of(NAME_KEY, name));
  }

  @Override
  public void notifyBeforeSuiteEnd(String name, String stackTrace, TestStatus status) {
    sendMessage(BEFORE_SUITE_END, of(NAME_KEY, name, STACK_TRACE_KEY, stackTrace, STATUS_KEY, status.toString()));
  }

  @Override
  public void notifyTestStart(String name) {
    sendMessage(TEST_START, of(NAME_KEY, name));
  }

  @Override
  public void notifyTestEnd(String name, String stackTrace, TestStatus status, long elapsedTime) {
    sendMessage(TEST_END, of(NAME_KEY, name,
                             STACK_TRACE_KEY, stackTrace,
                             ELAPSED_TIME_KEY, String.valueOf(elapsedTime),
                             STATUS_KEY, status.toString()));
  }

  @Override
  public void notifyAfterSuiteStart(String name) {
    sendMessage(AFTER_SUITE_START, of(NAME_KEY, name));
  }

  @Override
  public void notifyAfterSuiteEnd(String name, String stackTrace, TestStatus status) {
    sendMessage(AFTER_SUITE_END, of(NAME_KEY, name, STACK_TRACE_KEY, stackTrace, STATUS_KEY, status.toString()));
  }

  @Override
  public void notifySuiteEnd(String suite, long elapsedTime) {
    sendMessage(SUITE_END, of(MUNIT_SUITE_KEY, suite, ELAPSED_TIME_KEY, String.valueOf(elapsedTime)));
  }

  @Override
  public void notifyCoverageReport(String coverageReportJson) {
    sendMessage(COVERAGE_REPORT, of(COVERAGE_REPORT_KEY, coverageReportJson));
  }

  @Override
  public void notifyRunFinish() {
    sendMessage(RUN_FINISH, Collections.emptyMap());
  }

  @Override
  public void notifyUnexpectedError(String stackTrace) {
    sendMessage(UNEXPECTED_ERROR, of(STACK_TRACE_KEY, stackTrace));
  }

  @Override
  public void notifySuiteUnexpectedError(String stackTrace) {
    sendMessage(UNEXPECTED_SUITE_ERROR, of(STACK_TRACE_KEY, stackTrace));
  }

  private void sendMessage(Integer id, Map<String, String> parameters) {
    checkArgument(StringUtils.isNotBlank(runToken), "runToken must not be null nor empty");

    Map<String, String> completeParams = new HashMap<>();
    completeParams.put(RUN_TOKEN_KEY, runToken);
    completeParams.putAll(parameters);
    sendMessage(gson.toJson(new RunMessage(id, completeParams)));
  }

  public abstract void sendMessage(String message);

}
