/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.util;

import org.mule.munit.common.exception.MunitError;
import org.mule.runtime.api.el.ValidationResult;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.InternalEvent;
import org.mule.runtime.core.api.el.ExtendedExpressionManager;

// TODO improve implementation
/**
 * <p>
 * Provides a friendly API for the {@link ExtendedExpressionManager}
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MunitExpressionWrapper {

  private ExtendedExpressionManager expressionManager;

  public MunitExpressionWrapper(ExtendedExpressionManager expressionManager) {
    this.expressionManager = expressionManager;
  }

  public ExtendedExpressionManager getExpressionManager() {
    return this.expressionManager;
  }

  /**
   * It will try to evaluate the source sent, if it's a string and a valid expression.
   *
   * @param event the event over to which evaluate
   * @param source the variable containing the posible expression
   * @return the source if it's not an evaluable expression, or the result of the evaluation
   */
  public TypedValue evaluateIfExpression(Event event, Object source) {
    if (isExpression(source)) {
      return evaluate(event, source);
    }
    return new TypedValue<>(source, null);
  }


  public TypedValue evaluate(Event muleEvent, Object source) {
    validate(source);
    String stringSource = ((String) source);
    return expressionManager.evaluate(stringSource, (InternalEvent) muleEvent);
  }

  public boolean isExpression(Object source) {
    return null != source && source instanceof String && expressionManager.isExpression((String) source);
  }

  public boolean isExpressionValid(Object source) {
    if (!String.class.isAssignableFrom(source.getClass())) {
      return false;
    }
    String exp = (String) source;
    return expressionManager.isExpression(exp) && expressionManager.isValid(exp);

  }

  public void validate(Object expression) { // TODO shouldn't this be private
    if (expression instanceof String) {
      ValidationResult validationResult = expressionManager.validate((String) expression);
      if (!validationResult.isSuccess()) {
        throw new MunitError(
                             "Expression parsing failed. The expression: '" + expression + "' is invalid - " + validationResult
                                 .errorMessage());
      }

    } else {
      throw new MunitError("Expression parsing failed. The expression: '" + expression + "' is invalid (not a string)");
    }
  }

  /**
   * Tries to evaluate the given expression and cast it to String, if a java.lang.ClassCastException is thrown or the expression
   * evaluates to null it throws a MunitError
   *
   * @param event the event over to which evaluate
   * @param expression the expression to be evaluated
   * @param name the name of the variable to be evaluated
   * @return the string result from the expression evaluation
   */
  public String evaluateNotNullString(Event event, Object expression, String name) {
    String result = evaluateAsStringIfExpression(event, expression, name);
    if (result == null) {
      throw new MunitError(String.format("%s '%s' should not be null", name, expression));
    }
    return result;
  }


  /**
   * Tries to evaluate the given expression and cast it to String, if a java.lang.ClassCastException is thrown it throws a
   * MunitError
   *
   * @param event the event over to which evaluate
   * @param expression the expression to be evaluated
   * @return the string result from the expression evaluation
   */
  public String evaluateAsStringIfExpression(Event event, Object expression) {
    return evaluateAsStringIfExpression(event, expression, "Expression");
  }

  /**
   * Tries to evaluate the given expression and cast it to String, if a java.lang.ClassCastException is thrown it throws a
   * MunitError
   *
   * @param event the event over to which evaluate
   * @param expression the expression to be evaluated
   * @param name the name of the variable to be evaluated
   * @return the string result from the expression evaluation
   */
  // TODO it has to deal with typed value
  public String evaluateAsStringIfExpression(Event event, Object expression, String name) {
    try {
      return (String) evaluateIfExpression(event, expression).getValue();
    } catch (ClassCastException e) {
      throw new MunitError(String.format("%s '%s' should be evaluated as a String", name, expression), e);
    }
  }

}
