/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.util;

import static java.nio.charset.StandardCharsets.UTF_8;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.filefilter.FalseFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;

import java.io.*;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.security.SecureRandom;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Objects;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * <code>FileUtils</code> contains useful methods for dealing with files & directories. This is based on the mule core class so we
 * can have control over it
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class FileUtils extends org.apache.commons.io.FileUtils {

  public static final String DEFAULT_ENCODING = UTF_8.name();

  private static final SecureRandom random = new SecureRandom();

  /**
   * Remove from uri to file prefix file:/ Add if need file separator to begin
   *
   * @param url file uri to resource
   * @param encoding - Java encoding names
   * @return normalized file path
   * @throws java.io.UnsupportedEncodingException if encoding is unknown
   */
  public static String normalizeFilePath(URL url, String encoding) throws UnsupportedEncodingException {
    String resource = URLDecoder.decode(url.toExternalForm(), encoding);
    if (resource != null) {
      if (resource.startsWith("file:/")) {
        resource = resource.substring(6);

        if (!resource.startsWith(File.separator)) {
          resource = File.separator + resource;
        }
      }
    }
    return resource;
  }

  /**
   * Extract the specified resource to the given directory for remain all directory struct
   *
   * @param resourceName - full resource name
   * @param callingClass - classloader for this class is used
   * @param outputDir - extract to this directory
   * @param keepParentDirectory true - full structure of directories is kept; false - file - removed all directories, directory -
   *        started from resource point
   * @throws java.io.IOException if any errors
   */
  public static void extractResources(String resourceName, Class callingClass, File outputDir, boolean keepParentDirectory)
      throws IOException {
    URL url = callingClass.getClassLoader().getResource(resourceName);
    URLConnection connection = url.openConnection();
    if (connection instanceof JarURLConnection) {
      extractJarResources((JarURLConnection) connection, outputDir, keepParentDirectory);
    } else {
      extractFileResources(normalizeFilePath(url, DEFAULT_ENCODING),
                           outputDir, resourceName, keepParentDirectory);
    }
  }

  /**
   * Looks for a file inside a given folder
   *
   * @param folder Folder where to look for the file
   * @param filename Name of the file
   * @param recursive If true, the file will be looked for in all subfolders, otherwise it will be looked inside the given folder
   * @return The file in case it was found, null otherwise
   */
  public static File findFileByName(File folder, final String filename, boolean recursive) {
    Collection<File> files = findFiles(folder, new IOFileFilter() {

      @Override
      public boolean accept(File file) {
        return filename.equals(file.getName());
      }

      @Override
      public boolean accept(File dir, String name) {
        return true;
      }
    }, recursive);

    return CollectionUtils.isEmpty(files) ? null : files.iterator().next();
  }

  /**
   * Generates a file name appending a random number between the given prefix and suffix
   *
   * @param prefix prefix of the file name
   * @param suffix suffix of the file name
   * @return Generated File Name
   */
  public static String generateRandomFileName(String prefix, String suffix) {
    long n = random.nextLong();
    if (n == Long.MIN_VALUE) {
      n = 0;
    } else {
      n = Math.abs(n);
    }

    return prefix + Long.toString(n) + suffix;
  }

  /**
   * Check if a file or directory is inside a specific directory.
   * @param directory Base directory
   * @param file File to verify
   * @return True if the file exists inside the directory
   */
  public static boolean isInDirectory(File directory, File file) {
    if (Objects.isNull(directory) || !directory.isDirectory() || Objects.isNull(file)) {
      return false;
    }

    try {
      return file.getCanonicalPath().startsWith(directory.getCanonicalPath());
    } catch (IOException e) {
      return false;
    }
  }

  /**
   * Delete a file tree recursively.
   *
   * @param dir dir to wipe out
   * @return false when the first unsuccessful attempt encountered
   */
  static boolean deleteTree(File dir) {
    return deleteTree(dir, null);
  }

  /**
   * Delete a file tree recursively. This method additionally tries to be gentle with specified top-level dirs. E.g. this is the
   * case when a transaction manager asynchronously handles the recovery log, and the test wipes out everything, leaving the
   * transaction manager puzzled.
   *
   * @param dir dir to wipe out
   * @param topLevelDirsToIgnore which top-level directories to ignore, if null or empty then ignored
   * @return false when the first unsuccessful attempt encountered
   */
  static boolean deleteTree(File dir, final String[] topLevelDirsToIgnore) {
    if (dir == null || !dir.exists()) {
      return true;
    }
    File[] files = dir.listFiles();
    if (files != null) {
      for (int i = 0; i < files.length; i++) {
        OUTER: if (files[i].isDirectory()) {
          if (topLevelDirsToIgnore != null) {
            for (int j = 0; j < topLevelDirsToIgnore.length; j++) {
              String ignored = topLevelDirsToIgnore[j];
              if (ignored.equals(FilenameUtils.getBaseName(files[i].getName()))) {
                break OUTER;
              }
            }
          }
          if (!deleteTree(files[i])) {
            return false;
          }
        } else {
          if (!files[i].delete()) {
            return false;
          }
        }
      }
    }
    return dir.delete();
  }

  static File newFile(String pathName) {
    try {
      return new File(pathName).getCanonicalFile();
    } catch (IOException e) {
      throw new RuntimeException(String.format("Unable to create a canonical file for  %s", pathName), e);
    }
  }

  static File newFile(File parent, String child) {
    try {
      return new File(parent, child).getCanonicalFile();
    } catch (IOException e) {
      throw new RuntimeException("Unable to create a canonical file for parent: " + parent + " and child: " + child, e);
    }
  }

  static File newFile(String parent, String child) {
    try {
      return new File(parent, child).getCanonicalFile();
    } catch (IOException e) {
      throw new RuntimeException("Unable to create a canonical file for parent: " + parent + " and child: " + child, e);
    }
  }

  /**
   * Extract resources contain in file
   *
   * @param path - path to file
   * @param outputDir Directory for unpack recources
   * @param resourceName Name of the resource
   * @param keepParentDirectory true - full structure of directories is kept; false - file - removed all directories, directory -
   *        started from resource point
   * @throws java.io.IOException if any error
   */
  private static void extractFileResources(String path, File outputDir, String resourceName, boolean keepParentDirectory)
      throws IOException {
    File file = newFile(path);
    if (!file.exists()) {
      throw new IOException("The resource by path " + path + " ");
    }
    if (file.isDirectory()) {
      if (keepParentDirectory) {
        outputDir = newFile(outputDir.getPath() + File.separator + resourceName);
        if (!outputDir.exists()) {
          outputDir.mkdirs();
        }
      } else {
        outputDir = newFile(outputDir.getPath());
      }
      copyDirectory(file, outputDir);
    } else {

      if (keepParentDirectory) {
        outputDir = newFile(outputDir.getPath() + File.separator + resourceName);
      } else {
        outputDir = newFile(outputDir.getPath() + File.separator + file.getName());
      }
      copyFile(file, outputDir);
    }
  }

  /**
   * Extract recources contain if jar (have to in classpath)
   *
   * @param connection JarURLConnection to jar library
   * @param outputDir Directory for unpack recources
   * @param keepParentDirectory true - full structure of directories is kept; false - file - removed all directories, directory -
   *        started from resource point
   * @throws java.io.IOException if any error
   */
  private static void extractJarResources(JarURLConnection connection, File outputDir, boolean keepParentDirectory)
      throws IOException {
    JarFile jarFile = connection.getJarFile();
    JarEntry jarResource = connection.getJarEntry();
    Enumeration entries = jarFile.entries();
    InputStream inputStream = null;
    OutputStream outputStream = null;
    int jarResourceNameLenght = jarResource.getName().length();
    for (; entries.hasMoreElements();) {
      JarEntry entry = (JarEntry) entries.nextElement();
      if (entry.getName().startsWith(jarResource.getName())) {

        String path = outputDir.getPath() + File.separator + entry.getName();

        // remove directory struct for file and first dir for directory
        if (!keepParentDirectory) {
          if (entry.isDirectory()) {
            if (entry.getName().equals(jarResource.getName())) {
              continue;
            }
            path = outputDir.getPath() + File.separator + entry.getName()
                .substring(jarResourceNameLenght, entry.getName().length());
          } else {
            if (entry.getName().length() > jarResourceNameLenght) {
              path = outputDir.getPath() + File.separator + entry.getName()
                  .substring(jarResourceNameLenght, entry.getName().length());
            } else {
              path = outputDir.getPath() + File.separator + entry.getName()
                  .substring(entry.getName().lastIndexOf("/"), entry.getName().length());
            }
          }
        }

        File file = newFile(path);
        if (!file.getParentFile().exists()) {
          if (!file.getParentFile().mkdirs()) {
            throw new IOException("Could not create directory: " + file.getParentFile());
          }
        }
        if (entry.isDirectory()) {
          if (!file.exists() && !file.mkdirs()) {
            throw new IOException("Could not create directory: " + file);
          }

        } else {
          try {
            inputStream = jarFile.getInputStream(entry);
            outputStream = new BufferedOutputStream(new FileOutputStream(file));
            IOUtils.copy(inputStream, outputStream);
          } finally {
            IOUtils.closeQuietly(inputStream);
            IOUtils.closeQuietly(outputStream);
          }
        }

      }
    }
  }

  private static Collection<File> findFiles(File folder, IOFileFilter filter, boolean recursive) {
    return listFiles(folder, filter, recursive ? TrueFileFilter.INSTANCE : FalseFileFilter.INSTANCE);
  }


}
