/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.protocol.notifiers;

import static org.mule.munit.common.util.Preconditions.checkArgument;
import static org.mule.munit.common.protocol.message.MessageField.COVERAGE_REPORT_KEY;
import static org.mule.munit.common.protocol.message.MessageField.DESCRIPTION_KEY;
import static org.mule.munit.common.protocol.message.MessageField.ELAPSED_TIME_KEY;
import static org.mule.munit.common.protocol.message.MessageField.IGNORED_KEY;
import static org.mule.munit.common.protocol.message.MessageField.MUNIT_SUITE_KEY;
import static org.mule.munit.common.protocol.message.MessageField.NAME_KEY;
import static org.mule.munit.common.protocol.message.MessageField.NUMBER_OF_TESTS_KEY;
import static org.mule.munit.common.protocol.message.MessageField.PARAMETERIZATION_KEY;
import static org.mule.munit.common.protocol.message.MessageField.RUN_TOKEN_KEY;
import static org.mule.munit.common.protocol.message.MessageField.STACK_TRACE_KEY;
import static org.mule.munit.common.protocol.message.MessageField.STATUS_KEY;
import static org.mule.munit.common.protocol.message.MessageID.AFTER_SUITE_END;
import static org.mule.munit.common.protocol.message.MessageID.AFTER_SUITE_START;
import static org.mule.munit.common.protocol.message.MessageID.BEFORE_SUITE_END;
import static org.mule.munit.common.protocol.message.MessageID.BEFORE_SUITE_START;
import static org.mule.munit.common.protocol.message.MessageID.CONTAINER_START_FAILURE;
import static org.mule.munit.common.protocol.message.MessageID.CONTAINER_SUITE_START_FAILURE;
import static org.mule.munit.common.protocol.message.MessageID.COVERAGE_REPORT;
import static org.mule.munit.common.protocol.message.MessageID.RUN_FINISH;
import static org.mule.munit.common.protocol.message.MessageID.RUN_START;
import static org.mule.munit.common.protocol.message.MessageID.SUITE_END;
import static org.mule.munit.common.protocol.message.MessageID.SUITE_START;
import static org.mule.munit.common.protocol.message.MessageID.TEST_END;
import static org.mule.munit.common.protocol.message.MessageID.TEST_START;
import static org.mule.munit.common.protocol.message.MessageID.UNEXPECTED_ERROR;
import static org.mule.munit.common.protocol.message.MessageID.UNEXPECTED_SUITE_ERROR;
import static org.mule.munit.common.util.Collections.mapOf;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.message.RunMessage;
import org.mule.munit.common.protocol.message.TestStatus;

import com.google.gson.Gson;

/**
 * <p>
 * Abstract RunMessageNotifier class.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public abstract class RunMessageNotifier implements RunEventListener {

  protected String runToken;
  private Gson gson = new Gson();

  public abstract void init() throws Exception;

  public void setRunToken(String runToken) {
    this.runToken = runToken;
  }

  @Override
  public void notifyRunStart() {
    sendMessage(RUN_START, Collections.emptyMap());
  }

  @Override
  public void notifySuiteStart(String path, String parameterization, int numberOfTests) {
    sendMessage(SUITE_START,
                mapOf(MUNIT_SUITE_KEY, path, PARAMETERIZATION_KEY, nonNullValue(parameterization), NUMBER_OF_TESTS_KEY,
                      String.valueOf(numberOfTests)));
  }



  @Override
  public void notifyContainerFailure(String stackTrace) {
    sendMessage(CONTAINER_START_FAILURE, Collections.singletonMap(STACK_TRACE_KEY, stackTrace));
  }

  @Override
  public void notifyContainerFailure(String suite, String parameterization, String stackTrace) {
    sendMessage(CONTAINER_SUITE_START_FAILURE,
                mapOf(MUNIT_SUITE_KEY, suite, PARAMETERIZATION_KEY, nonNullValue(parameterization), STACK_TRACE_KEY, stackTrace));
  }

  @Override
  public void notifyBeforeSuiteStart(String name) {
    sendMessage(BEFORE_SUITE_START, Collections.singletonMap(NAME_KEY, name));
  }

  @Override
  public void notifyBeforeSuiteEnd(String name, String stackTrace, TestStatus status) {
    sendMessage(BEFORE_SUITE_END, mapOf(NAME_KEY, name, STACK_TRACE_KEY, stackTrace, STATUS_KEY, status.toString()));
  }

  @Override
  public void notifyTestStart(String name, String description, boolean isIgnored) {
    sendMessage(TEST_START, mapOf(NAME_KEY, name, DESCRIPTION_KEY, description, IGNORED_KEY, String.valueOf(isIgnored)));
  }

  @Override
  public void notifyTestEnd(String name, String stackTrace, TestStatus status, long elapsedTime) {
    sendMessage(TEST_END, mapOf(NAME_KEY, name,
                                STACK_TRACE_KEY, stackTrace,
                                ELAPSED_TIME_KEY, String.valueOf(elapsedTime),
                                STATUS_KEY, status.toString()));
  }

  @Override
  public void notifyAfterSuiteStart(String name) {
    sendMessage(AFTER_SUITE_START, Collections.singletonMap(NAME_KEY, name));
  }

  @Override
  public void notifyAfterSuiteEnd(String name, String stackTrace, TestStatus status) {
    sendMessage(AFTER_SUITE_END, mapOf(NAME_KEY, name, STACK_TRACE_KEY, stackTrace, STATUS_KEY, status.toString()));
  }

  @Override
  public void notifySuiteEnd(String suite, String parameterization, long elapsedTime) {
    sendMessage(SUITE_END,
                mapOf(MUNIT_SUITE_KEY, suite, PARAMETERIZATION_KEY, nonNullValue(parameterization), ELAPSED_TIME_KEY,
                      String.valueOf(elapsedTime)));
  }

  @Override
  public void notifyCoverageReport(String coverageReportJson) {
    sendMessage(COVERAGE_REPORT, Collections.singletonMap(COVERAGE_REPORT_KEY, coverageReportJson));
  }

  @Override
  public void notifyRunFinish() {
    sendMessage(RUN_FINISH, Collections.emptyMap());
  }

  @Override
  public void notifyUnexpectedError(String stackTrace) {
    sendMessage(UNEXPECTED_ERROR, Collections.singletonMap(STACK_TRACE_KEY, stackTrace));
  }

  @Override
  public void notifySuiteUnexpectedError(String name, String stackTrace) {
    sendMessage(UNEXPECTED_SUITE_ERROR, mapOf(NAME_KEY, name, STACK_TRACE_KEY, stackTrace));
  }

  private void sendMessage(Integer id, Map<String, String> parameters) {
    checkArgument(StringUtils.isNotBlank(runToken), "runToken must not be null nor empty");

    Map<String, String> completeParams = new HashMap<>();
    completeParams.put(RUN_TOKEN_KEY, runToken);
    completeParams.putAll(parameters);
    sendMessage(gson.toJson(new RunMessage(id, completeParams)));
  }

  private String nonNullValue(String value) {
    return value == null ? StringUtils.EMPTY : value;
  }

  public abstract void sendMessage(String message);

}
