/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.exception;

import java.util.Objects;

/**
 * <p>
 * Format a message using the expected and actual values
 * </p>
 * @author Mulesoft Inc.
 * @since 2.3.17
 */
public class ComparisonCompactor {

  private static String format(String message, Object expected, Object actual) {
    String formatted = "";
    if (message != null && message.length() > 0) {
      formatted = message + " ";
    }
    return formatted + "expected:<" + expected + "> but was:<" + actual + ">";
  }

  private static final String ELLIPSIS = "...";
  private static final String DELTA_END = "]";
  private static final String DELTA_START = "[";

  private final int contextLength;
  private final String expected;
  private final String actual;
  private int prefix;
  private int suffix;

  public ComparisonCompactor(int contextLength, String expected, String actual) {
    this.contextLength = contextLength;
    this.expected = expected;
    this.actual = actual;
  }

  public String compact(String message) {
    if (expected == null || actual == null || areStringsEqual()) {
      return format(message, expected, actual);
    }

    findCommonPrefix();
    findCommonSuffix();
    String expected = compactString(this.expected);
    String actual = compactString(this.actual);
    return format(message, expected, actual);
  }

  private String compactString(String source) {
    String result = DELTA_START + source.substring(prefix, source.length() - suffix + 1) + DELTA_END;
    if (prefix > 0) {
      result = computeCommonPrefix() + result;
    }
    if (suffix > 0) {
      result = result + computeCommonSuffix();
    }
    return result;
  }

  private void findCommonPrefix() {
    prefix = 0;
    int end = Math.min(expected.length(), actual.length());
    for (; prefix < end; prefix++) {
      if (expected.charAt(prefix) != actual.charAt(prefix)) {
        break;
      }
    }
  }

  private void findCommonSuffix() {
    int expectedSuffix = expected.length() - 1;
    int actualSuffix = actual.length() - 1;
    for (; actualSuffix >= prefix && expectedSuffix >= prefix; actualSuffix--, expectedSuffix--) {
      if (expected.charAt(expectedSuffix) != actual.charAt(actualSuffix)) {
        break;
      }
    }
    suffix = expected.length() - expectedSuffix;
  }

  private String computeCommonPrefix() {
    return (prefix > contextLength ? ELLIPSIS : "") + expected.substring(Math.max(0, prefix - contextLength), prefix);
  }

  private String computeCommonSuffix() {
    int end = Math.min(expected.length() - suffix + 1 + contextLength, expected.length());
    return expected.substring(expected.length() - suffix + 1, end)
        + (expected.length() - suffix + 1 < expected.length() - contextLength ? ELLIPSIS : "");
  }

  private boolean areStringsEqual() {
    return Objects.equals(expected, actual);
  }
}
