/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.api;/*
                                           * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
                                           * copyright law. All use of this software is subject to MuleSoft's Master Subscription
                                           * Agreement (or other master license agreement) separately entered into in writing between
                                           * you and MuleSoft. If such an agreement is not in place, you may not use the software.
                                           */

import static java.lang.String.format;
import static java.nio.file.Files.readAllBytes;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.io.FileUtils.readFileToString;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.maven.model.Resource;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.mule.munit.extension.maven.internal.generator.ApplicationMuleArtifactJsonGenerator;
import org.mule.munit.extension.maven.internal.generator.ApplicationPomGenerator;
import org.mule.munit.extension.maven.internal.generator.ExtensionApplicationStructureGenerator;
import org.mule.munit.extension.maven.internal.generator.SharedLibrariesPluginPomEnricher;
import org.mule.munit.extension.maven.internal.locator.ExportedPackagesLocator;
import org.mule.munit.extension.maven.internal.locator.ExportedResourcesLocator;
import org.mule.munit.plugin.maven.AbstractMunitMojo;
import org.mule.munit.plugin.maven.TargetRuntime;
import org.mule.munit.plugin.maven.locators.TestConfigFilesLocator;
import org.mule.munit.plugin.maven.util.BaseRunConfigurationFactory;
import org.mule.munit.plugin.maven.util.DefaultResultPrinterFactory;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.tools.api.classloader.model.SharedLibraryDependency;

/**
 * Runs MUnit tests over an extension
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
@Mojo(name = "test", defaultPhase = LifecyclePhase.INTEGRATION_TEST, requiresDependencyResolution = ResolutionScope.TEST)
public class ExtensionTestMojo extends AbstractMunitMojo {

  protected static final String TEMPORAL_EXTENSION_MODEL_JSON = "temporal-extension-model.json";

  @Parameter(defaultValue = "${project.build.directory}")
  protected File buildDirectory;

  @Parameter(defaultValue = "${project.build.outputDirectory}/META-INF/mule-artifact/mule-artifact.json")
  protected File muleArtifactJsonFile;

  @Parameter(defaultValue = "${project.basedir}/src/test/java")
  protected File testSourcesDirectory;

  @Parameter(defaultValue = "${project.testResources}")
  protected List<Resource> testResources;

  @Parameter(property = "shared.libraries")
  protected List<SharedLibraryDependency> sharedLibraries;

  @Parameter(property = "mule.packager.version", defaultValue = "3.1.0")
  protected String packagerVersion;

  @Parameter(property = "targetRuntimes")
  protected List<String> targetRuntimes;

  protected ApplicationMuleArtifactJsonGenerator muleArtifactJsonGenerator;

  protected ApplicationPomGenerator applicationPomGenerator;

  private ExtensionModel extensionModel;

  @Override
  protected void init() throws MojoExecutionException {
    muleArtifactJsonGenerator =
        new ApplicationMuleArtifactJsonGenerator(getConfigs(), getExportedResources(), getExportedPackages());

    extensionModel = getExtensionModel();
    applicationPomGenerator = new ApplicationPomGenerator(project, getSharedLibrariesEnricher(), pluginArtifactId);
    super.init();
  }

  @Override
  protected Map<TargetRuntime, RunConfiguration> getRunConfigurations() throws MojoExecutionException {
    Map<TargetRuntime, RunConfiguration> runConfigurations = new HashMap<>();
    if (targetRuntimes == null || targetRuntimes.isEmpty()) {
      TargetRuntime targetRuntime =
          new TargetRuntime(muleApplicationModelLoader.getRuntimeVersion(), muleApplicationModelLoader.getRuntimeProduct());
      runConfigurations.put(targetRuntime, createRunConfiguration(targetRuntime));
    } else {
      for (String targetRuntimeCoordinate : targetRuntimes) {
        TargetRuntime targetRuntime = TargetRuntime.parse(targetRuntimeCoordinate);
        runConfigurations.put(targetRuntime, createRunConfiguration(targetRuntime));
      }
    }
    return runConfigurations;
  }

  @Override
  protected ResultPrinterFactory getResultPrinterFactory() {
    return new DefaultResultPrinterFactory(getLog())
        .withSurefireReports(enableSurefireReports, surefireReportsFolder, effectiveSystemProperties)
        .withTestOutputReports(redirectTestOutputToFile, testOutputDirectory);
  }

  @Override
  protected ExtensionApplicationStructureGenerator getApplicationStructureGenerator() throws MojoExecutionException {
    try {
      return new ExtensionApplicationStructureGenerator(buildDirectory.toPath(), project.getGroupId(),
                                                        project.getArtifactId(),
                                                        generatePomXml(), generatePomProperties(),
                                                        generateMuleArtifactJson());
    } catch (IOException e) {
      throw new MojoExecutionException("Unable to create application structure", e);
    }
  }

  @Override
  protected File getMuleApplicationJsonPath() {
    return muleArtifactJsonFile;
  }

  protected byte[] generateMuleArtifactJson() throws IOException {
    String muleArtifactJson = readFileToString(muleArtifactJsonFile, Charset.defaultCharset());
    MuleApplicationModel originalMuleApplicationModel = new MuleApplicationModelJsonSerializer().deserialize(muleArtifactJson);
    MuleApplicationModel appMuleApplicationModel = muleArtifactJsonGenerator.generate(originalMuleApplicationModel);
    return new MuleApplicationModelJsonSerializer().serialize(appMuleApplicationModel).getBytes(Charset.defaultCharset());
  }

  protected byte[] generatePomXml() throws IOException {
    ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
    new MavenXpp3Writer().write(byteArrayOutputStream, applicationPomGenerator.generate());
    return byteArrayOutputStream.toByteArray();
  }

  protected byte[] generatePomProperties() throws IOException {
    Properties properties = new Properties();
    properties.setProperty("groupId", project.getGroupId());
    properties.setProperty("artifactId", project.getArtifactId());
    properties.setProperty("version", project.getVersion());
    ByteArrayOutputStream output = new ByteArrayOutputStream();
    properties.store(output, "Generated by " + pluginArtifactId);
    return output.toByteArray();
  }

  private RunConfiguration createRunConfiguration(TargetRuntime targetRuntime)
      throws MojoExecutionException {
    return new BaseRunConfigurationFactory(getLog(), munitTest, munitTags, skipAfterFailure, targetRuntime,
                                           workingDirectoryGenerator, munitTestsDirectory, project, session).create();
  }

  private Set<String> getConfigs() {
    return new TestConfigFilesLocator().locateFiles(munitTestsDirectory)
        .stream()
        .map(suiteFile -> munitTestsDirectory.toURI().relativize(suiteFile.toURI()))
        .map(URI::getPath).collect(toSet());
  }

  private Set<String> getExportedResources() {
    return testResources.stream().map(resource -> new ExportedResourcesLocator(new File(resource.getDirectory())).locate())
        .flatMap(Set::stream)
        .collect(toSet());
  }

  private Set<String> getExportedPackages() {
    return new ExportedPackagesLocator(testSourcesDirectory).locate();
  }

  private SharedLibrariesPluginPomEnricher getSharedLibrariesEnricher() throws MojoExecutionException {
    return new SharedLibrariesPluginPomEnricher(getExtensionModel(), getSharedLibraries(), packagerVersion);
  }

  private List<SharedLibraryDependency> getSharedLibraries() {
    return sharedLibraries == null ? emptyList() : sharedLibraries;
  }

  protected ExtensionModel getExtensionModel() throws MojoExecutionException {
    if (extensionModel == null) {
      final File generatedExtensionModelFile = new File(buildDirectory, TEMPORAL_EXTENSION_MODEL_JSON);
      try {
        final String serializedExModel = new String(readAllBytes(generatedExtensionModelFile.toPath()), "UTF-8");
        extensionModel = new ExtensionModelJsonSerializer(true).deserialize(serializedExModel);
      } catch (Exception e) {
        throw new MojoExecutionException(
                                         format("Cannot obtain/read the file [%s] to deserialize the ExtensionModel",
                                                generatedExtensionModelFile.getAbsolutePath()),
                                         e);
      }
    }

    return extensionModel;
  }

}
