/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.internal.generator;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.munit.remote.FolderNames.CLASSES;
import static org.mule.munit.remote.FolderNames.MAVEN;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.munit.remote.FolderNames.MUNIT;
import static org.mule.munit.remote.FolderNames.TEST_CLASSES;
import static org.mule.munit.remote.FolderNames.TEST_MULE;
import static org.mule.tools.api.packager.structure.PackagerFiles.MULE_ARTIFACT_JSON;
import static org.mule.tools.api.packager.structure.PackagerFiles.POM_PROPERTIES;
import static org.mule.tools.api.packager.structure.PackagerFiles.POM_XML;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;

import org.apache.commons.io.FileUtils;
import org.mule.munit.remote.api.project.ApplicationStructureGenerator;
import org.mule.tools.api.packager.archiver.MuleExplodedArchiver;

import com.google.common.io.Files;

/**
 * Creates an application structure based on an extensions build directory
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class ExtensionApplicationStructureGenerator implements ApplicationStructureGenerator {

  private static final String[] CLASSES_EXCLUSIONS =
      {META_INF.value() + File.separator + MULE_ARTIFACT.value() + File.separator + MULE_ARTIFACT_JSON};

  private Path originFolder;
  private byte[] pomXmlContent;
  private byte[] pomPropertiesContent;
  private byte[] muleArtifactJson;
  private String groupId;
  private String artifactId;
  private File tempDir;

  public ExtensionApplicationStructureGenerator(Path originFolder, String groupId, String artifactId, byte[] pomXmlContent,
                                                byte[] pomPropertiesContent, byte[] muleArtifactJsonContent) {
    checkArgument(originFolder != null, "Origin folder cannot be null");
    checkArgument(originFolder.toFile().exists(), "Origin folder should exist");
    checkArgument(isNotBlank(groupId), "Group Id cannot be blank");
    checkArgument(isNotBlank(artifactId), "Artifact Id cannot be blank");
    checkArgument(pomXmlContent != null, "Pom file content cannot be null");
    checkArgument(pomPropertiesContent != null, "Pom properties content cannot be null");
    checkArgument(muleArtifactJsonContent != null, "Mule Artifact Json content cannot be null");

    this.originFolder = originFolder;
    this.pomXmlContent = pomXmlContent;
    this.pomPropertiesContent = pomPropertiesContent;
    this.muleArtifactJson = muleArtifactJsonContent;
    this.groupId = groupId;
    this.artifactId = artifactId;
    this.tempDir = Files.createTempDir();
  }

  @Override
  public Path generate(Path destinationFolder) throws Exception {
    checkArgument(destinationFolder != null, "Destination folder cannot be null");

    MuleExplodedArchiver archiver = new MuleExplodedArchiver();

    addResources(archiver);
    addPomDescriptors(archiver);
    addMuleArtifactJson(archiver);

    archiver.setDestFile(destinationFolder.toFile());
    archiver.createArchive();

    FileUtils.cleanDirectory(tempDir);
    return destinationFolder;
  }

  private void addMuleArtifactJson(MuleExplodedArchiver archiver) throws IOException {
    File muleArtifactFolder = new File(tempDir, MULE_ARTIFACT.value());
    FileUtils.copyToFile(new ByteArrayInputStream(muleArtifactJson), new File(muleArtifactFolder, MULE_ARTIFACT_JSON));

    archiver.addMuleArtifact(muleArtifactFolder, null, null);
  }

  private void addPomDescriptors(MuleExplodedArchiver archiver) throws IOException {
    File mavenFolder = new File(Files.createTempDir(), MAVEN.value());
    File groupIdFolder = new File(mavenFolder, groupId);
    File artifactIdFolder = new File(groupIdFolder, artifactId);
    FileUtils.copyToFile(new ByteArrayInputStream(pomXmlContent), new File(artifactIdFolder, POM_XML));
    FileUtils.copyToFile(new ByteArrayInputStream(pomPropertiesContent), new File(artifactIdFolder, POM_PROPERTIES));

    archiver.addMaven(mavenFolder, null, null);
  }

  private void addResources(MuleExplodedArchiver archiver) {
    addToRootIfExists(archiver, originFolder.resolve(CLASSES.value()).toFile(), null, CLASSES_EXCLUSIONS);
    addToRootIfExists(archiver, originFolder.resolve(TEST_CLASSES.value()).toFile(), null, null);
    addToRootIfExists(archiver, originFolder.resolve(TEST_MULE.value()).resolve(MUNIT.value()).toFile(), null, null);
  }

  private void addToRootIfExists(MuleExplodedArchiver archiver, File file, String[] includes, String[] excludes) {
    if (file.exists()) {
      archiver.addToRoot(file, includes, excludes);
    }
  }
}
