/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.api;

import static java.util.Optional.ofNullable;

import java.util.Optional;
import java.util.Properties;

import org.mule.munit.plugin.maven.ProductDiscoveryMode;

/**
 * Model of the {@link RuntimeConfiguration} discoveryRuntimes parameter. It holds information about the runtimes discovery
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class DiscoverRuntimes {

  public static final String DISCOVER_RUNTIMES_PRODUCT = "discoverRuntimes.product";
  public static final String DISCOVER_RUNTIMES_INCLUDE_SNAPSHOTS = "discoverRuntimes.includeSnapshots";
  public static final String DISCOVER_RUNTIMES_MIN_MULE_VERSION = "discoverRuntimes.minMuleVersion";
  public static final String DISCOVER_RUNTIMES_SKIP = "discoverRuntimes.skip";

  private ProductDiscoveryMode product;
  private String minMuleVersion;
  private boolean includeSnapshots = false;
  private boolean skip = false;

  public DiscoverRuntimes() {}

  private DiscoverRuntimes(ProductDiscoveryMode product, String minMuleVersion, boolean includeSnapshots, boolean skip) {
    this.product = product;
    this.minMuleVersion = minMuleVersion;
    this.includeSnapshots = includeSnapshots;
    this.skip = skip;
  }

  public Optional<String> getMinMuleVersion() {
    return ofNullable(minMuleVersion);
  }

  public Optional<ProductDiscoveryMode> getProduct() {
    return ofNullable(product);
  }

  public boolean isIncludeSnapshots() {
    return includeSnapshots;
  }

  public boolean isSkipped() {
    return skip;
  }

  public static Builder builder() {
    return new Builder();
  }

  static class Builder {

    private ProductDiscoveryMode product;
    private String minMuleVersion;
    private boolean includeSnapshots = false;
    private boolean skip;

    Builder from(DiscoverRuntimes discoverRuntimes) {
      if (discoverRuntimes == null) {
        this.skip = true;
        return this;
      }
      discoverRuntimes.getMinMuleVersion().ifPresent(minMuleVersion -> this.minMuleVersion = minMuleVersion);
      discoverRuntimes.getProduct().ifPresent(productDiscoveryMode -> this.product = productDiscoveryMode);
      this.includeSnapshots = discoverRuntimes.isIncludeSnapshots();
      this.skip = discoverRuntimes.isSkipped();
      return this;
    }

    Builder withProduct(ProductDiscoveryMode productDiscoveryMode) {
      this.product = productDiscoveryMode;
      return this;
    }

    Builder withMinMuleVersion(String minMuleVersion) {
      this.minMuleVersion = minMuleVersion;
      return this;
    }

    Builder includingSnapshots(Boolean includeSnapshots) {
      this.includeSnapshots = includeSnapshots;
      return this;
    }

    Builder skip(Boolean skip) {
      this.skip = skip;
      return this;
    }

    Builder withOverrides(Properties userProperties) {
      if (userProperties.containsKey(DISCOVER_RUNTIMES_INCLUDE_SNAPSHOTS)) {
        this.includeSnapshots = Boolean.valueOf(userProperties.getProperty(DISCOVER_RUNTIMES_INCLUDE_SNAPSHOTS));
      }
      if (userProperties.containsKey(DISCOVER_RUNTIMES_PRODUCT)) {
        this.product = ProductDiscoveryMode.valueOf(userProperties.getProperty(DISCOVER_RUNTIMES_PRODUCT));
      }
      if (userProperties.containsKey(DISCOVER_RUNTIMES_MIN_MULE_VERSION)) {
        this.minMuleVersion = userProperties.getProperty(DISCOVER_RUNTIMES_MIN_MULE_VERSION);
      }
      if (userProperties.containsKey(DISCOVER_RUNTIMES_SKIP)) {
        this.skip = Boolean.valueOf(userProperties.getProperty(DISCOVER_RUNTIMES_SKIP));
      }
      return this;
    }

    DiscoverRuntimes build() {
      return new DiscoverRuntimes(product, minMuleVersion, includeSnapshots, skip);
    }

  }

}
