/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.api;/*
                                           * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
                                           * copyright law. All use of this software is subject to MuleSoft's Master Subscription
                                           * Agreement (or other master license agreement) separately entered into in writing between
                                           * you and MuleSoft. If such an agreement is not in place, you may not use the software.
                                           */

import static java.lang.String.format;
import static java.nio.file.Files.readAllBytes;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections.CollectionUtils.isNotEmpty;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.mule.munit.extension.maven.api.DiscoverRuntimes.DISCOVER_RUNTIMES_PRODUCT;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.maven.model.Resource;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;

import org.mule.munit.extension.maven.internal.generator.ApplicationMuleArtifactJsonGenerator;
import org.mule.munit.extension.maven.internal.generator.ExtensionApplicationStructureGenerator;
import org.mule.munit.extension.maven.internal.generator.MuleSourcesApplicationGenerator;
import org.mule.munit.extension.maven.internal.generator.maven.ApplicationPomGenerator;
import org.mule.munit.extension.maven.internal.generator.maven.MuleMavenPluginEnricher;
import org.mule.munit.extension.maven.internal.generator.maven.MunitMavenPluginEnricher;
import org.mule.munit.extension.maven.internal.generator.maven.PomEnricher;
import org.mule.munit.extension.maven.internal.locator.ExportedPackagesLocator;
import org.mule.munit.extension.maven.internal.locator.ExportedResourcesLocator;
import org.mule.munit.plugin.maven.AbstractMunitMojo;
import org.mule.munit.plugin.maven.ProductDiscoveryMode;
import org.mule.munit.plugin.maven.TargetRuntime;
import org.mule.munit.plugin.maven.locators.TestConfigFilesLocator;
import org.mule.munit.plugin.maven.util.BaseRunConfigurationFactory;
import org.mule.munit.plugin.maven.util.DefaultResultPrinterFactory;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.tools.api.classloader.model.SharedLibraryDependency;

/**
 * Runs MUnit tests over an extension
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
@Mojo(name = "test", defaultPhase = LifecyclePhase.INTEGRATION_TEST, requiresDependencyResolution = ResolutionScope.TEST)
public class ExtensionTestMojo extends AbstractMunitMojo {

  protected static final String TEMPORAL_EXTENSION_MODEL_JSON = "temporal-extension-model.json";

  @Parameter(defaultValue = "${project.build.directory}")
  protected File buildDirectory;

  @Parameter(defaultValue = "${project.build.outputDirectory}/META-INF/mule-artifact/mule-artifact.json")
  protected File muleArtifactJsonFile;

  @Parameter(defaultValue = "${project.basedir}/src/test/java")
  protected File testSourcesDirectory;

  @Parameter(defaultValue = "${project.testResources}")
  protected List<Resource> testResources;

  @Parameter(property = "shared.libraries")
  protected List<SharedLibraryDependency> sharedLibraries;

  @Parameter(property = "mule.packager.version", defaultValue = "3.1.2")
  protected String packagerVersion;

  @Parameter(property = "munit.maven.plugin.version", defaultValue = "2.1.2")
  protected String munitMavenPluginVersion;

  @Parameter
  protected RuntimeConfiguration runtimeConfiguration;

  @Parameter(property = "attachMuleSources", defaultValue = "false")
  protected boolean attachMuleSources;

  @Parameter(property = "testApplicationName", defaultValue = "${project.artifactId}-test-application")
  protected String testApplicationName;

  protected ApplicationMuleArtifactJsonGenerator muleArtifactJsonGenerator;
  protected ApplicationPomGenerator applicationPomGenerator;

  private ExtensionModel extensionModel;

  @Override
  protected void init() throws MojoExecutionException {
    muleArtifactJsonGenerator =
        new ApplicationMuleArtifactJsonGenerator(getConfigs(), getExportedResources(), getExportedPackages());

    extensionModel = getExtensionModel();
    applicationPomGenerator = new ApplicationPomGenerator(project, getPomEnrichers(), testApplicationName, pluginArtifactId);
    super.init();
    workingDirectoryGenerator.setDestinationName(testApplicationName);
    runtimeConfiguration =
        RuntimeConfiguration.builder().from(runtimeConfiguration).withOverrides(session.getUserProperties()).build();
  }

  @Override
  protected Map<TargetRuntime, RunConfiguration> getRunConfigurations() throws MojoExecutionException {
    Map<TargetRuntime, RunConfiguration> runConfigurations = new HashMap<>();
    List<TargetRuntime> effectiveTargetRuntimes = new ArrayList<>();

    addDefaultTargetRuntime(effectiveTargetRuntimes);
    if (!shouldRunSingleRuntime()) {
      List<String> additionalRuntimes = runtimeConfiguration.getAdditionalRuntimes();
      Optional<DiscoverRuntimes> discoverRuntimes = runtimeConfiguration.getDiscoverRuntimes();
      boolean shouldDiscoverRuntimes = discoverRuntimes.map(dr -> !dr.isSkipped()).orElse(false);
      if (isNotEmpty(additionalRuntimes)) {
        effectiveTargetRuntimes.addAll(additionalRuntimes.stream().map(TargetRuntime::parse).collect(Collectors.toList()));
      }
      if (shouldDiscoverRuntimes) {
        effectiveTargetRuntimes.addAll(discoverRuntimes(muleApplicationModelLoader.getRuntimeVersion(), discoverRuntimes.get()));
      }
    }
    for (TargetRuntime targetRuntime : effectiveTargetRuntimes) {
      runConfigurations.put(targetRuntime, createRunConfiguration(targetRuntime));
    }
    return runConfigurations;
  }

  @Override
  protected ResultPrinterFactory getResultPrinterFactory() {
    return new DefaultResultPrinterFactory(getLog())
        .withSurefireReports(enableSurefireReports, surefireReportsFolder, effectiveSystemProperties)
        .withTestOutputReports(redirectTestOutputToFile, testOutputDirectory);
  }

  @Override
  protected ExtensionApplicationStructureGenerator getApplicationStructureGenerator() throws MojoExecutionException {
    try {
      return new ExtensionApplicationStructureGenerator(buildDirectory.toPath(), project.getGroupId(),
                                                        testApplicationName, generatePomXml(), generatePomProperties(),
                                                        generateMuleArtifactJson());
    } catch (IOException e) {
      throw new MojoExecutionException("Unable to create application structure", e);
    }
  }

  @Override
  protected File getMuleApplicationJsonPath() {
    return muleArtifactJsonFile;
  }

  @Override
  public void doExecute() throws MojoExecutionException {
    generateMuleSources();
    super.doExecute();
  }

  protected byte[] generateMuleArtifactJson() throws IOException {
    String muleArtifactJson = readFileToString(muleArtifactJsonFile, Charset.defaultCharset());
    MuleApplicationModel originalMuleApplicationModel = new MuleApplicationModelJsonSerializer().deserialize(muleArtifactJson);
    MuleApplicationModel appMuleApplicationModel = muleArtifactJsonGenerator.generate(originalMuleApplicationModel);
    return new MuleApplicationModelJsonSerializer().serialize(appMuleApplicationModel).getBytes(Charset.defaultCharset());
  }

  protected byte[] generatePomXml() throws IOException {
    ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
    new MavenXpp3Writer().write(byteArrayOutputStream, applicationPomGenerator.generate());
    return byteArrayOutputStream.toByteArray();
  }

  protected byte[] generatePomProperties() throws IOException {
    Properties properties = new Properties();
    properties.setProperty("groupId", project.getGroupId());
    properties.setProperty("artifactId", testApplicationName);
    properties.setProperty("version", project.getVersion());
    ByteArrayOutputStream output = new ByteArrayOutputStream();
    properties.store(output, "Generated by " + pluginArtifactId);
    return output.toByteArray();
  }

  private Set<TargetRuntime> discoverRuntimes(String runtimeVersion, DiscoverRuntimes discoverRuntimes)
      throws MojoExecutionException {
    runtimeVersionsLocator
        .includingSnapshots(discoverRuntimes.isIncludeSnapshots())
        .withMinMuleVersion(discoverRuntimes.getMinMuleVersion().orElse(runtimeVersion))
        .withProductDiscoveryMode(discoverRuntimes.getProduct()
            .orElseThrow(() -> new MojoExecutionException(format("Invalid Runtime Configuration: the auto discovery of runtimes was enabled but no product kind was specified, neither in the plugin nor using the command line argument '%s'."
                + " One of the following products has to be selected: %s", DISCOVER_RUNTIMES_PRODUCT,
                                                                 asList(ProductDiscoveryMode.values())))));
    return runtimeVersionsLocator.locate();
  }

  private void addDefaultTargetRuntime(List<TargetRuntime> effectiveTargetRuntimes) {
    effectiveTargetRuntimes
        .add(new TargetRuntime(muleApplicationModelLoader.getRuntimeVersion(), muleApplicationModelLoader.getRuntimeProduct()));
  }

  private RunConfiguration createRunConfiguration(TargetRuntime targetRuntime)
      throws MojoExecutionException {
    return new BaseRunConfigurationFactory(getLog(), testApplicationName, munitTest, munitTags, skipAfterFailure, targetRuntime,
                                           workingDirectoryGenerator, munitTestsDirectory, project, session).create();
  }

  private Set<String> getConfigs() {
    return new TestConfigFilesLocator().locateFiles(munitTestsDirectory)
        .stream()
        .map(suiteFile -> munitTestsDirectory.toURI().relativize(suiteFile.toURI()))
        .map(URI::getPath).collect(toSet());
  }

  private Set<String> getExportedResources() {
    return testResources.stream().map(resource -> new ExportedResourcesLocator(new File(resource.getDirectory())).locate())
        .flatMap(Set::stream)
        .collect(toSet());
  }

  private Set<String> getExportedPackages() {
    return new ExportedPackagesLocator(testSourcesDirectory).locate();
  }

  private List<PomEnricher> getPomEnrichers() throws MojoExecutionException {
    List<PomEnricher> pomEnrichers = new ArrayList<>();
    pomEnrichers.add(new MuleMavenPluginEnricher(getExtensionModel(), project, getSharedLibraries(), packagerVersion));
    pomEnrichers.add(new MunitMavenPluginEnricher(munitMavenPluginVersion));
    return pomEnrichers;
  }

  private List<SharedLibraryDependency> getSharedLibraries() {
    return sharedLibraries == null ? emptyList() : sharedLibraries;
  }

  private void generateMuleSources() throws MojoExecutionException {
    if (attachMuleSources) {
      new MuleSourcesApplicationGenerator(testApplicationName, project, getApplicationStructureGenerator())
          .generate(buildDirectory.toPath());
    }
  }

  private boolean shouldRunSingleRuntime() {
    return runtimeVersion != null || runtimeProduct != null;
  }

  protected ExtensionModel getExtensionModel() throws MojoExecutionException {
    if (extensionModel == null) {
      final File generatedExtensionModelFile = new File(buildDirectory, TEMPORAL_EXTENSION_MODEL_JSON);
      try {
        final String serializedExModel = new String(readAllBytes(generatedExtensionModelFile.toPath()), "UTF-8");
        extensionModel = new ExtensionModelJsonSerializer(true).deserialize(serializedExModel);
      } catch (Exception e) {
        throw new MojoExecutionException(
                                         format("Cannot obtain/read the file [%s] to deserialize the ExtensionModel",
                                                generatedExtensionModelFile.getAbsolutePath()),
                                         e);
      }
    }

    return extensionModel;
  }

}
