/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.internal.generator.maven;

import static java.util.Arrays.asList;

import java.io.File;
import java.util.List;

import org.apache.maven.model.Dependency;
import org.apache.maven.model.Model;
import org.apache.maven.model.Plugin;
import org.apache.maven.project.MavenProject;

/**
 * Generates an application {@link Model} based on an extension's {@link org.apache.maven.project.MavenProject}
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class ApplicationPomGenerator {

  protected static final String APPLICATION_PACKAGING = "mule-application";
  protected static final String MULE_PLUGIN_CLASSIFIER = "mule-plugin";
  protected static final String EXTENSION_DEPENDENCY_SCOPE = "system";
  protected static final String MODEL_VERSION = "4.0.0";
  protected static final String TEST_APPLICATION_SUFFIX = "-test-application";

  public static final String TEST_SCOPE = "test";

  private static final List<String> APP_INCLUDED_SCOPES = asList(TEST_SCOPE);
  private static final List<DependencyExclusion> BLACKLISTED_DEPENDENCIES =
      asList(new DependencyExclusion("org.mule.tests.plugin", "mule-tests-component-plugin"));
  private final List<PomEnricher> pomEnrichers;

  private String projectArtifactId;
  private MavenProject project;
  private String pluginArtifactId;

  public ApplicationPomGenerator(MavenProject project, List<PomEnricher> pomEnrichers, String projectArtifactId,
                                 String pluginArtifactId) {
    this.project = project;
    this.projectArtifactId = projectArtifactId;
    this.pluginArtifactId = pluginArtifactId;
    this.pomEnrichers = pomEnrichers;
  }

  public Model generate() {
    Model pomModel = new Model();
    setGeneralAttributes(pomModel);
    addDependencies(pomModel);
    addPomEnrichers(pomModel);
    return pomModel;
  }

  private void setGeneralAttributes(Model pomModel) {
    pomModel.setGroupId(project.getGroupId());
    pomModel.setArtifactId(projectArtifactId);
    pomModel.setVersion(project.getVersion());
    pomModel.setPackaging(APPLICATION_PACKAGING);
    pomModel.setModelVersion(MODEL_VERSION);
  }

  private void addDependencies(Model pomModel) {
    addMUnitPluginDependencies(pomModel);
    addExtensionAsDependency(pomModel);
    addFilteredDependencies(pomModel);
  }

  private void addMUnitPluginDependencies(Model pomModel) {
    Plugin munitPlugin = project.getBuild().getPlugins().stream()
        .filter(p -> p.getArtifactId().equals(pluginArtifactId))
        .findAny()
        .orElseThrow(() -> new IllegalStateException("No [" + pluginArtifactId + "] defined in the extension pom.xml"));

    List<Dependency> dependencies = munitPlugin.getDependencies();
    if (dependencies != null) {
      dependencies.forEach(dependency -> {
        Dependency newDependency = dependency.clone();
        newDependency.setScope(TEST_SCOPE);
        pomModel.addDependency(newDependency);
      });
    }
  }

  private void addExtensionAsDependency(Model pomModel) {
    Dependency dependency = new Dependency();
    dependency.setGroupId(project.getGroupId());
    dependency.setArtifactId(project.getArtifactId());
    dependency.setVersion(project.getVersion());
    dependency.setSystemPath(getExtensionJarPath());
    dependency.setScope(EXTENSION_DEPENDENCY_SCOPE);
    dependency.setClassifier(MULE_PLUGIN_CLASSIFIER);
    pomModel.addDependency(dependency);
  }

  /**
   * Looks for the extension jar file that follows the [${baseDirectory}/target/artifactId-version-classifier.jar] name
   * convention.
   *
   * @return the absolute path of the jar file, if exists.
   * @throws IllegalStateException if the extension jar file was not found.
   */
  private String getExtensionJarPath() {
    String buildDir = project.getBuild().getDirectory();
    String path = buildDir + "/" + project.getArtifactId() + "-" + project.getVersion() + "-" + MULE_PLUGIN_CLASSIFIER + ".jar";
    if (!new File(path).exists()) {
      throw new IllegalStateException("Could not find module jar file, [" + path + "] does not exist.");
    }
    return path;
  }

  private void addPomEnrichers(Model pomModel) {
    pomEnrichers.forEach(pomEnricher -> pomEnricher.generate(pomModel));
  }

  private void addFilteredDependencies(Model pomModel) {
    project.getDependencies().stream()
        .filter(dependency -> APP_INCLUDED_SCOPES.contains(dependency.getScope()))
        .filter(this::isNotBlacklisted)
        .forEach(pomModel::addDependency);
  }

  private boolean isNotBlacklisted(Dependency dependency) {
    return BLACKLISTED_DEPENDENCIES.stream().noneMatch(exclusion -> exclusion.excludes(dependency));
  }

  private static class DependencyExclusion {

    private String groupId;
    private String artifactId;

    DependencyExclusion(String groupId, String artifactId) {
      this.groupId = groupId;
      this.artifactId = artifactId;
    }

    boolean excludes(Dependency dependency) {
      return groupId.equals(dependency.getGroupId()) && artifactId.equals(dependency.getArtifactId());
    }
  }

}
