/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.internal.generator.maven;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.ExternalLibraryModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.util.IdempotentExtensionWalker;
import org.mule.tools.api.classloader.model.SharedLibraryDependency;

import com.google.common.collect.ImmutableList;

import java.util.List;

import org.apache.maven.model.Dependency;
import org.apache.maven.model.Model;
import org.apache.maven.model.Plugin;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Enriches a given {@link Model POM} with a {@link Plugin mule-maven-plugin} declaration configured with {@code sharedLibraries}
 * that applies for the given extension project.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class MuleMavenPluginEnricher extends PluginPomEnricher {

  private static final Logger LOGGER = LoggerFactory.getLogger(MuleMavenPluginEnricher.class);

  public static final String MAVEN_TOOLS_GROUP_ID = "org.mule.tools.maven";
  public static final String MULE_MAVEN_PLUGIN_ARTIFACT_ID = "mule-maven-plugin";
  public static final String GROUP_ID = "groupId";
  public static final String ARTIFACT_ID = "artifactId";
  public static final String SHARED_LIBRARY_ELEMENT = "sharedLibrary";
  public static final String SHARED_LIBRARIES_ELEMENT = "sharedLibraries";
  public static final String COMPILE_SCOPE = "compile";

  private final List<SharedLibraryDependency> parameterizedSharedLibraries;
  private final String packagerVersion;
  private final ExtensionModel extensionModel;
  private final MavenProject mavenProject;

  public MuleMavenPluginEnricher(ExtensionModel extensionModel, MavenProject mavenProject,
                                 List<SharedLibraryDependency> sharedLibraries, String packagerVersion) {
    this.extensionModel = extensionModel;
    this.packagerVersion = packagerVersion;
    this.mavenProject = mavenProject;
    this.parameterizedSharedLibraries = sharedLibraries;
  }

  /**
   * Enriches a given {@link Model POM} with a {@link Plugin mule-maven-plugin} declaration configured with
   * {@code sharedLibraries} that applies for the given extension project.
   */
  @Override
  public void generate(Model pomModel) {
    Xpp3Dom configuration = createConfiguration(pomModel);
    Plugin muleMavenPlugin =
        createPlugin(MAVEN_TOOLS_GROUP_ID, MULE_MAVEN_PLUGIN_ARTIFACT_ID, packagerVersion, configuration, emptyList(), true);
    getBuildNode(pomModel).addPlugin(muleMavenPlugin);
  }

  private Xpp3Dom createConfiguration(Model pomModel) {
    List<ExternalLibraryModel> externalLibs = getExternalLibs(extensionModel);
    if (externalLibs.isEmpty()) {
      return null;
    }

    validateParameterizedLibraries(externalLibs);
    if (parameterizedSharedLibraries.isEmpty()) {
      return null;
    }

    List<Dependency> sharedLibrariesDependencies = getSharedLibrariesDependencies(mavenProject.getDependencies());
    addSharedLibrariesDependencies(sharedLibrariesDependencies, pomModel);
    final Xpp3Dom sharedLibraries = createSharedLibrariesDeclarations(sharedLibrariesDependencies);
    return createConfiguration(sharedLibraries);
  }

  private List<Dependency> getSharedLibrariesDependencies(List<Dependency> projectDependencies) {
    return projectDependencies.stream().filter(this::isDeclaredAsSharedLib).collect(toList());
  }

  private void addSharedLibrariesDependencies(List<Dependency> sharedLibrariesDependencies, Model pomModel) {
    sharedLibrariesDependencies.stream().filter(sharedLibDep -> isNotAlreadyPresent(pomModel, sharedLibDep))
        .map(this::asCompileDependency).forEach(pomModel::addDependency);
  }

  private boolean isNotAlreadyPresent(Model pomModel, Dependency sharedLibDependency) {
    return pomModel.getDependencies().stream().noneMatch(dep -> dep.getGroupId().equals(sharedLibDependency.getGroupId())
        && dep.getArtifactId().equals(sharedLibDependency.getArtifactId()));
  }

  private Xpp3Dom createSharedLibrariesDeclarations(List<Dependency> sharedLibsDependencies) {
    final Xpp3Dom sharedLibraries = new Xpp3Dom(SHARED_LIBRARIES_ELEMENT);
    sharedLibsDependencies.stream()
        .filter(this::isDeclaredAsSharedLib)
        .forEach(dependency -> sharedLibraries.addChild(createSharedLibraryDeclaration(dependency)));
    return sharedLibraries;
  }

  private void validateParameterizedLibraries(List<ExternalLibraryModel> externalLibs) {
    List<String> requiredLibs = externalLibs.stream()
        .filter(l -> !l.isOptional())
        .map(ExternalLibraryModel::getName)
        .collect(toList());

    if (parameterizedSharedLibraries.size() < requiredLibs.size()) {
      StringBuilder sb = new StringBuilder()
          .append("Missing SharedLibraries configuration. ")
          .append("There are [").append(requiredLibs.size()).append("] required libraries ");

      if (parameterizedSharedLibraries.isEmpty()) {
        sb.append("but none was declared.");
      } else {
        sb.append("but only [").append(parameterizedSharedLibraries.size()).append("] ")
            .append(parameterizedSharedLibraries.size() > 1 ? "were" : "was")
            .append(" declared.");

      }

      sb.append("\n");

      sb.append("Required libraries are: ").append(requiredLibs).append(".");
      if (!parameterizedSharedLibraries.isEmpty()) {
        sb.append("\n").append("Provided libraries artifact-ids are: ")
            .append(parameterizedSharedLibraries.stream().map(SharedLibraryDependency::getArtifactId).collect(toList()));
      }

      sb.append("\n").append("Some tests may fail due to this missing libraries");

      LOGGER.warn(sb.toString());
    }
  }

  private List<ExternalLibraryModel> getExternalLibs(ExtensionModel extensionModel) {
    ImmutableList.Builder<ExternalLibraryModel> libs = ImmutableList.<ExternalLibraryModel>builder()
        .addAll(extensionModel.getExternalLibraryModels());

    new IdempotentExtensionWalker() {

      @Override
      protected void onConnectionProvider(ConnectionProviderModel model) {
        libs.addAll(model.getExternalLibraryModels());
      }

      @Override
      protected void onConfiguration(ConfigurationModel model) {
        libs.addAll(model.getExternalLibraryModels());
      }
    }.walk(extensionModel);

    return libs.build();
  }

  private Xpp3Dom createSharedLibraryDeclaration(Dependency dependency) {
    final Xpp3Dom groupId = new Xpp3Dom(GROUP_ID);
    groupId.setValue(dependency.getGroupId());
    final Xpp3Dom artifactId = new Xpp3Dom(ARTIFACT_ID);
    artifactId.setValue(dependency.getArtifactId());

    final Xpp3Dom sharedLibrary = new Xpp3Dom(SHARED_LIBRARY_ELEMENT);
    sharedLibrary.addChild(groupId);
    sharedLibrary.addChild(artifactId);
    return sharedLibrary;
  }

  private boolean isDeclaredAsSharedLib(Dependency dep) {
    return parameterizedSharedLibraries.stream()
        .anyMatch(lib -> lib.getGroupId().equals(dep.getGroupId()) && lib.getArtifactId().equals(dep.getArtifactId()));
  }

  private Dependency asCompileDependency(Dependency originalDependency) {
    Dependency dependency = originalDependency.clone();
    dependency.setScope(COMPILE_SCOPE);
    return dependency;
  }

}
