/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.api;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.split;

import java.util.List;
import java.util.Optional;
import java.util.Properties;

import org.mule.munit.plugin.maven.ProductDiscoveryMode;

/**
 * Model of the {@link RuntimeConfiguration} discoveryRuntimes parameter. It holds information about the runtimes discovery
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class DiscoverRuntimes {

  public static final String DISCOVER_RUNTIMES_PRODUCT = "discoverRuntimes.product";
  public static final String DISCOVER_RUNTIMES_INCLUDE_SNAPSHOTS = "discoverRuntimes.includeSnapshots";
  public static final String DISCOVER_RUNTIMES_MIN_MULE_VERSION = "discoverRuntimes.minMuleVersion";
  public static final String DISCOVER_RUNTIMES_SKIP = "discoverRuntimes.skip";
  public static final String DISCOVER_RUNTIMES_REPOSITORIES = "discoverRuntimes.runtimeRepositories";
  private static final String DISCOVER_RUNTIMES_REPOSITORIES_SEPARATOR = ",";

  private ProductDiscoveryMode product;
  private String minMuleVersion;
  private boolean includeSnapshots = false;
  private boolean skip = false;
  private List<String> runtimeRepositories = emptyList();

  public DiscoverRuntimes() {}

  private DiscoverRuntimes(ProductDiscoveryMode product, String minMuleVersion, boolean includeSnapshots, boolean skip,
                           List<String> runtimeRepositories) {
    this.product = product;
    this.minMuleVersion = minMuleVersion;
    this.includeSnapshots = includeSnapshots;
    this.skip = skip;
    this.runtimeRepositories = runtimeRepositories;
  }

  public Optional<String> getMinMuleVersion() {
    return ofNullable(minMuleVersion);
  }

  public Optional<ProductDiscoveryMode> getProduct() {
    return ofNullable(product);
  }

  public boolean isIncludeSnapshots() {
    return includeSnapshots;
  }

  public boolean isSkipped() {
    return skip;
  }

  public List<String> getRuntimeRepositories() {
    return runtimeRepositories;
  }

  public static Builder builder() {
    return new Builder();
  }

  static class Builder {

    private ProductDiscoveryMode product;
    private String minMuleVersion;
    private boolean includeSnapshots = false;
    private boolean skip;
    private List<String> runtimeRepositories = emptyList();

    Builder from(DiscoverRuntimes discoverRuntimes) {
      if (discoverRuntimes == null) {
        this.skip = true;
        return this;
      }
      discoverRuntimes.getMinMuleVersion().ifPresent(minMuleVersion -> this.minMuleVersion = minMuleVersion);
      discoverRuntimes.getProduct().ifPresent(productDiscoveryMode -> this.product = productDiscoveryMode);
      this.includeSnapshots = discoverRuntimes.isIncludeSnapshots();
      this.skip = discoverRuntimes.isSkipped();
      this.runtimeRepositories = discoverRuntimes.getRuntimeRepositories();
      return this;
    }

    Builder withProduct(ProductDiscoveryMode productDiscoveryMode) {
      this.product = productDiscoveryMode;
      return this;
    }

    Builder withMinMuleVersion(String minMuleVersion) {
      this.minMuleVersion = minMuleVersion;
      return this;
    }

    Builder includingSnapshots(Boolean includeSnapshots) {
      this.includeSnapshots = includeSnapshots;
      return this;
    }

    Builder skip(Boolean skip) {
      this.skip = skip;
      return this;
    }

    Builder withRuntimeRepositories(List<String> runtimeRepositories) {
      this.runtimeRepositories = runtimeRepositories;
      return this;
    }

    Builder withOverrides(Properties userProperties) {
      if (userProperties.containsKey(DISCOVER_RUNTIMES_INCLUDE_SNAPSHOTS)) {
        this.includeSnapshots = Boolean.valueOf(userProperties.getProperty(DISCOVER_RUNTIMES_INCLUDE_SNAPSHOTS));
      }
      if (userProperties.containsKey(DISCOVER_RUNTIMES_PRODUCT)) {
        this.product = ProductDiscoveryMode.valueOf(userProperties.getProperty(DISCOVER_RUNTIMES_PRODUCT));
      }
      if (userProperties.containsKey(DISCOVER_RUNTIMES_MIN_MULE_VERSION)) {
        this.minMuleVersion = userProperties.getProperty(DISCOVER_RUNTIMES_MIN_MULE_VERSION);
      }
      if (userProperties.containsKey(DISCOVER_RUNTIMES_SKIP)) {
        this.skip = Boolean.valueOf(userProperties.getProperty(DISCOVER_RUNTIMES_SKIP));
      }
      if (userProperties.containsKey(DISCOVER_RUNTIMES_REPOSITORIES)) {
        this.runtimeRepositories = asList(split(userProperties.getProperty(DISCOVER_RUNTIMES_REPOSITORIES),
                                                DISCOVER_RUNTIMES_REPOSITORIES_SEPARATOR));
      }
      return this;
    }

    DiscoverRuntimes build() {
      return new DiscoverRuntimes(product, minMuleVersion, includeSnapshots, skip, runtimeRepositories);
    }

  }

}
