/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.api;

import static java.lang.String.format;
import static java.nio.file.Files.readAllBytes;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.io.FileUtils.readFileToString;
import org.mule.munit.extension.maven.internal.generator.ApplicationMuleArtifactJsonGenerator;
import org.mule.munit.extension.maven.internal.generator.ApplicationRepositoryGenerator;
import org.mule.munit.extension.maven.internal.generator.ExtensionApplicationStructureGenerator;
import org.mule.munit.extension.maven.internal.generator.MuleSourcesApplicationGenerator;
import org.mule.munit.extension.maven.internal.generator.maven.ApplicationPomGenerator;
import org.mule.munit.extension.maven.internal.generator.maven.CloudHubDependencyGenerator;
import org.mule.munit.extension.maven.internal.generator.maven.DependencyGenerator;
import org.mule.munit.extension.maven.internal.generator.maven.EmbeddedDependencyGenerator;
import org.mule.munit.extension.maven.internal.generator.maven.MuleMavenPluginEnricher;
import org.mule.munit.extension.maven.internal.generator.maven.MunitMavenPluginEnricher;
import org.mule.munit.extension.maven.internal.generator.maven.PomEnricher;
import org.mule.munit.extension.maven.internal.locator.ExportedPackagesLocator;
import org.mule.munit.extension.maven.internal.locator.ExportedResourcesLocator;
import org.mule.munit.plugin.maven.AbstractMunitMojo;
import org.mule.munit.plugin.maven.locators.RemoteRepositoriesLocator;
import org.mule.munit.plugin.maven.locators.TestConfigFilesLocator;
import org.mule.munit.plugin.maven.runtime.TargetRuntime;
import org.mule.munit.plugin.maven.util.BaseApplicationRunConfigurationFactory;
import org.mule.munit.plugin.maven.util.BaseCloudHubRunConfigurationFactory;
import org.mule.munit.plugin.maven.util.DefaultResultPrinterFactory;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.tools.client.BAT.BATClient;
import org.mule.munit.remote.tools.client.BAT.model.request.TestType;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.tools.api.classloader.model.SharedLibraryDependency;
import org.mule.tools.api.util.MavenComponents;
import org.mule.tools.client.authentication.AuthenticationServiceClient;
import org.mule.tools.client.authentication.model.Credentials;
import org.mule.tools.model.anypoint.CloudHubDeployment;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;

/**
 * Runs MUnit tests over an extension
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
@Mojo(name = "test", defaultPhase = LifecyclePhase.INTEGRATION_TEST, requiresDependencyResolution = ResolutionScope.TEST)
public class ExtensionTestMojo extends AbstractMunitMojo {

  protected static final String TEMPORAL_EXTENSION_MODEL_JSON = "temporal-extension-model.json";

  @Parameter(defaultValue = "${project.build.directory}")
  protected File buildDirectory;

  @Parameter(defaultValue = "${project.build.outputDirectory}/META-INF/mule-artifact/mule-artifact.json")
  protected File muleArtifactJsonFile;

  @Parameter(defaultValue = "${project.build.directory}/test-classes")
  protected File testClassesDirectory;

  @Parameter(defaultValue = "${project.basedir}/src/test/java")
  protected File testSourcesDirectory;

  @Parameter(defaultValue = "${project.testResources}")
  protected List<Resource> testResources;

  @Parameter(property = "shared.libraries")
  protected List<SharedLibraryDependency> sharedLibraries;

  @Parameter(property = "mule.packager.version", defaultValue = "3.1.2")
  protected String packagerVersion;

  @Parameter(property = "munit.maven.plugin.version", defaultValue = "2.1.2")
  protected String munitMavenPluginVersion;

  @Parameter
  protected Set<String> externalSuites = emptySet();

  @Parameter
  protected Set<String> externalResources = emptySet();

  @Parameter(property = "attachMuleSources", defaultValue = "false")
  protected boolean attachMuleSources;

  @Parameter(property = "testApplicationName", defaultValue = "${project.artifactId}-test-application")
  protected String testApplicationName;

  protected ApplicationMuleArtifactJsonGenerator muleArtifactJsonGenerator;
  protected ApplicationPomGenerator applicationPomGenerator;

  private ExtensionModel extensionModel;

  @Override
  protected void init() throws MojoExecutionException {
    copyExternalFilesToCurrentProject();
    muleArtifactJsonGenerator =
        new ApplicationMuleArtifactJsonGenerator(getConfigs(), getExportedResources(), getExportedPackages());
    extensionModel = getExtensionModel();

    DependencyGenerator dependencyGenerator = createDependencyGenerator();
    applicationPomGenerator =
        new ApplicationPomGenerator(project, getPomEnrichers(), testApplicationName, pluginArtifactId, getLog(),
                                    dependencyGenerator);
    super.init();
    workingDirectoryGenerator.setDestinationName(testApplicationName);
  }

  @Override
  protected ResultPrinterFactory getResultPrinterFactory() {
    return new DefaultResultPrinterFactory(getLog())
        .withSurefireReports(enableSurefireReports, surefireReportsFolder, effectiveSystemProperties)
        .withTestOutputReports(redirectTestOutputToFile, testOutputDirectory);
  }

  @Override
  protected ExtensionApplicationStructureGenerator getApplicationStructureGenerator() throws MojoExecutionException {
    try {
      ExtensionApplicationStructureGenerator applicationGenerator =
          new ExtensionApplicationStructureGenerator(buildDirectory.toPath(), project.getGroupId(),
                                                     testApplicationName, applicationPomGenerator, generatePomProperties(),
                                                     generateMuleArtifactJson());
      if (super.cloudHubDeployment != null) {
        MavenComponents mavenComponents = new MavenComponents()
            .withLog(getLog())
            .withOutputDirectory(outputDirectory)
            .withSession(session)
            .withProjectBuilder(projectBuilder)
            .withRepositorySystem(mavenRepositorySystem)
            .withLocalRepository(localRepository)
            .withRemoteArtifactRepositories(remoteArtifactRepositories)
            .withClassifier(StringUtils.EMPTY)
            .withProjectBaseFolder(projectBaseFolder)
            .withSharedLibraries(getSharedLibraries());
        applicationGenerator.withRepositoryGenerator(new ApplicationRepositoryGenerator(mavenComponents, project));
      }
      return applicationGenerator;
    } catch (IOException e) {
      throw new MojoExecutionException("Unable to create application structure", e);
    }
  }

  @Override
  protected Map<TargetRuntime, RunConfiguration> getRunConfigurations() throws MojoExecutionException {
    Map<TargetRuntime, RunConfiguration> runConfigurations = super.getRunConfigurations();
    for (RunConfiguration runConfiguration : runConfigurations.values()) {
      generateMuleSources(runConfiguration);
    }
    return runConfigurations;
  }

  @Override
  protected File getMuleApplicationJsonPath() {
    return muleArtifactJsonFile;
  }

  @Override
  protected RemoteRepositoriesLocator createRemoteRepositoriesLocator() {
    RemoteRepositoriesLocator remoteRepositoriesLocator = super.createRemoteRepositoriesLocator();
    if (runtimeConfiguration != null) {
      return runtimeConfiguration.getDiscoverRuntimes()
          .map(discoverRuntimes -> remoteRepositoriesLocator.setWhiteListRepositories(discoverRuntimes.getRuntimeRepositories()))
          .orElse(remoteRepositoriesLocator);
    } else {
      return remoteRepositoriesLocator;
    }
  }

  protected byte[] generateMuleArtifactJson() throws IOException {
    String muleArtifactJson = readFileToString(muleArtifactJsonFile, Charset.defaultCharset());
    MuleApplicationModel originalMuleApplicationModel = new MuleApplicationModelJsonSerializer().deserialize(muleArtifactJson);
    MuleApplicationModel appMuleApplicationModel = muleArtifactJsonGenerator.generate(originalMuleApplicationModel);
    return new MuleApplicationModelJsonSerializer().serialize(appMuleApplicationModel).getBytes(Charset.defaultCharset());
  }

  protected byte[] generatePomProperties() throws IOException {
    Properties properties = new Properties();
    properties.setProperty("groupId", project.getGroupId());
    properties.setProperty("artifactId", testApplicationName);
    properties.setProperty("version", project.getVersion());
    ByteArrayOutputStream output = new ByteArrayOutputStream();
    properties.store(output, "Generated by " + pluginArtifactId);
    return output.toByteArray();
  }

  @Override
  protected RunConfiguration createRunConfiguration(TargetRuntime targetRuntime)
      throws MojoExecutionException {
    if (cloudHubDeployment != null) {
      return new BaseCloudHubRunConfigurationFactory(getLog(), testApplicationName, munitTest, munitTags, skipAfterFailure,
                                                     targetRuntime, workingDirectoryGenerator, munitTestsDirectory, project,
                                                     session, cloudHubDeployment, systemPropertyVariables, environmentVariables,
                                                     getExecutionId(cloudHubDeployment))
                                                         .create();
    } else {
      return new BaseApplicationRunConfigurationFactory(getLog(), testApplicationName, munitTest, munitTags, skipAfterFailure,
                                                        targetRuntime, workingDirectoryGenerator, munitTestsDirectory, project,
                                                        session).create();
    }
  }

  private String getExecutionId(CloudHubDeployment deployment) {
    BATClient client = new BATClient(deployment.getUri(), new Credentials(deployment.getUsername(), deployment.getPassword()),
                                     new AuthenticationServiceClient(deployment.getUri(), true));
    return client.createExecution(TestType.MUNIT).getExecutionId();
  }

  private Set<String> getConfigs() {
    return new TestConfigFilesLocator().locateFiles(munitTestsDirectory)
        .stream()
        .map(suiteFile -> munitTestsDirectory.toURI().relativize(suiteFile.toURI()))
        .map(URI::getPath).collect(toSet());
  }

  private Set<String> getExportedResources() {
    return testResources.stream().map(resource -> new ExportedResourcesLocator(new File(resource.getDirectory())).locate())
        .flatMap(Set::stream)
        .collect(toSet());
  }

  private Set<String> getExportedPackages() {
    return new ExportedPackagesLocator(testSourcesDirectory).locate();
  }

  private List<PomEnricher> getPomEnrichers() {
    List<PomEnricher> pomEnrichers = new ArrayList<>();
    pomEnrichers
        .add(new MuleMavenPluginEnricher(extensionModel, project, getSharedLibraries(), pluginArtifactId, packagerVersion));
    pomEnrichers.add(new MunitMavenPluginEnricher(munitMavenPluginVersion));
    return pomEnrichers;
  }

  private List<SharedLibraryDependency> getSharedLibraries() {
    return sharedLibraries == null ? emptyList() : sharedLibraries;
  }

  private void generateMuleSources(RunConfiguration runConfiguration) throws MojoExecutionException {
    if (attachMuleSources) {
      new MuleSourcesApplicationGenerator(testApplicationName, project, getApplicationStructureGenerator())
          .generate(buildDirectory.toPath(), runConfiguration);
    }
  }

  protected ExtensionModel getExtensionModel() throws MojoExecutionException {
    if (extensionModel == null) {
      final File generatedExtensionModelFile = new File(buildDirectory, TEMPORAL_EXTENSION_MODEL_JSON);
      if (!generatedExtensionModelFile.exists()) {
        getLog().debug(TEMPORAL_EXTENSION_MODEL_JSON + " file not found in " + buildDirectory);
        return null;
      }
      try {
        final String serializedExModel = new String(readAllBytes(generatedExtensionModelFile.toPath()), "UTF-8");
        extensionModel = new ExtensionModelJsonSerializer(true).deserialize(serializedExModel);
      } catch (Exception e) {
        throw new MojoExecutionException(format("Cannot obtain/read the file [%s] to deserialize the ExtensionModel",
                                                generatedExtensionModelFile.getAbsolutePath()),
                                         e);
      }
    }

    return extensionModel;
  }

  protected void copyExternalFilesToCurrentProject() throws MojoExecutionException {
    copyExternalResources(externalSuites, munitTestsDirectory);
    copyExternalResources(externalResources, testClassesDirectory);
  }

  private void copyExternalResources(Collection<String> resources, File outputDirectory) throws MojoExecutionException {
    for (String resource : resources) {
      InputStream suiteContent = Thread.currentThread().getContextClassLoader().getResourceAsStream(resource);
      if (suiteContent != null) {
        File suiteFile = new File(outputDirectory, resource);
        try {
          FileUtils.copyInputStreamToFile(suiteContent, suiteFile);
        } catch (Exception e) {
          throw new MojoExecutionException(format("Resource [%s] could not be loaded", resource), e);
        }
      }
    }
  }

  private DependencyGenerator createDependencyGenerator() {
    if (cloudHubDeployment != null) {
      return new CloudHubDependencyGenerator();
    } else {
      return new EmbeddedDependencyGenerator();
    }
  }
}
