/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.internal.generator.maven;

import static java.util.Arrays.asList;
import static org.mule.munit.common.util.VersionUtils.isAtLeastMinMuleVersion;
import static org.mule.tools.api.packager.structure.FolderNames.META_INF;
import static org.mule.tools.api.packager.structure.FolderNames.MULE_ARTIFACT;
import static org.mule.tools.api.packager.structure.PackagerFiles.MULE_ARTIFACT_JSON;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;
import org.mule.runtime.core.internal.util.JarUtils;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Build;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Model;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

/**
 * Generates an application {@link Model} based on an extension's {@link org.apache.maven.project.MavenProject}
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class ApplicationPomGenerator {

  protected static final String APPLICATION_PACKAGING = "mule-application";
  protected static final String MULE_PLUGIN_CLASSIFIER = "mule-plugin";
  protected static final String MODEL_VERSION = "4.0.0";

  public static final String TEST_SCOPE = "test";

  private static final List<String> APP_INCLUDED_SCOPES = asList(TEST_SCOPE);
  private static final List<DependencyExclusion> BLACKLISTED_DEPENDENCIES =
      asList(new ArtifactExclusion("org.mule.tests.plugin", "mule-tests-component-plugin"),
             new ArtifactExclusion("org.mule.distributions", "mule-distro-tests"),
             new ClassifierExclusion("mule-service"));

  private static final String MULE_ARTIFACT_PATH_INSIDE_JAR = META_INF.value() + "/" + MULE_ARTIFACT.value();
  private static final String MULE_ARTIFACT_JSON_IN_JAR = MULE_ARTIFACT_PATH_INSIDE_JAR + "/" + MULE_ARTIFACT_JSON;
  private static final String MTF_TOOLS_MIN_MULE_VERSION = "4.2.0";
  private static final String MTF_TOOLS_GROUP_ID = "com.mulesoft.munit";
  private static final String MTF_TOOLS_ARTIFACT_ID = "mtf-tools";

  private final List<PomEnricher> pomEnrichers;

  private final String projectArtifactId;
  private final MavenProject project;
  private final String pluginArtifactId;
  private final Log log;
  private final DependencyGenerator dependencyGenerator;

  public ApplicationPomGenerator(MavenProject project, List<PomEnricher> pomEnrichers, String projectArtifactId,
                                 String pluginArtifactId, Log log, DependencyGenerator dependencyGenerator) {
    this.project = project;
    this.projectArtifactId = projectArtifactId;
    this.pluginArtifactId = pluginArtifactId;
    this.pomEnrichers = pomEnrichers;
    this.log = log;
    this.dependencyGenerator = dependencyGenerator;
  }

  public Model generate(RunConfiguration runConfiguration) {
    Model pomModel = new Model();
    setGeneralAttributes(pomModel);
    addDependencies(pomModel, runConfiguration);
    Build build = new Build();
    build.setDirectory(project.getBuild().getDirectory());
    pomModel.setBuild(build);
    addPomEnrichers(pomModel);
    return pomModel;
  }

  private void setGeneralAttributes(Model pomModel) {
    pomModel.setGroupId(project.getGroupId());
    pomModel.setArtifactId(projectArtifactId);
    pomModel.setVersion(project.getVersion());
    pomModel.setPackaging(APPLICATION_PACKAGING);
    pomModel.setModelVersion(MODEL_VERSION);
  }

  private void addDependencies(Model pomModel, RunConfiguration runConfiguration) {
    addMUnitPluginDependencies(pomModel, runConfiguration);
    addExtensionAsDependency(pomModel);
    addFilteredDependencies(pomModel, runConfiguration);
  }

  private void addMUnitPluginDependencies(Model pomModel, RunConfiguration runConfiguration) {
    List<Dependency> dependencies = ProjectUtils.getPluginDependencies(project, pluginArtifactId);
    if (dependencies != null) {
      dependencies.stream().filter(dep -> isMtfToolsSupported(dep, runConfiguration)).forEach(dependency -> {
        Dependency newDependency = dependency.clone();
        newDependency.setScope(TEST_SCOPE);
        pomModel.addDependency(newDependency);
      });
    }
  }

  private boolean isMtfToolsSupported(Dependency dependency, RunConfiguration runConfiguration) {
    if (dependency.getGroupId().equals(MTF_TOOLS_GROUP_ID) && dependency.getArtifactId().equals(MTF_TOOLS_ARTIFACT_ID)) {
      String runtimeVersion = runConfiguration.getContainerConfiguration().getRuntimeId();
      boolean isAtLeastMtfToolsMinVersion = isAtLeastMinMuleVersion(runtimeVersion, MTF_TOOLS_MIN_MULE_VERSION);
      if (!isAtLeastMtfToolsMinVersion) {
        log.debug("Excluded dependency " + dependency.getArtifactId() + " since it won't run against runtime " + runtimeVersion);
      }
      return isAtLeastMtfToolsMinVersion;
    }
    return true;
  }

  private void addExtensionAsDependency(Model pomModel) {
    pomModel.addDependency(dependencyGenerator.generateDependency(project));
  }

  private void addPomEnrichers(Model pomModel) {
    pomEnrichers.forEach(pomEnricher -> pomEnricher.generate(pomModel));
  }

  private void addFilteredDependencies(Model pomModel, RunConfiguration runConfiguration) {
    project.getDependencies().stream()
        .filter(dependency -> APP_INCLUDED_SCOPES.contains(dependency.getScope()))
        .filter(this::isNotBlacklisted)
        .filter(minMuleVersionCompatible(runConfiguration))
        .forEach(pomModel::addDependency);
  }

  private Predicate<Dependency> minMuleVersionCompatible(RunConfiguration runConfiguration) {
    return dependency -> isMinMuleVersionCompatible(runConfiguration, dependency);
  }

  private boolean isMinMuleVersionCompatible(RunConfiguration runConfiguration, Dependency dependency) {
    String runtimeVersion = runConfiguration.getContainerConfiguration().getRuntimeId();
    boolean dependencyMinMuleVersionCompatible = isDependencyMinMuleVersionCompatible(dependency, runtimeVersion);
    if (!dependencyMinMuleVersionCompatible) {
      log.debug("Excluded dependency " + dependency.getArtifactId() + " since it won't run against runtime " + runtimeVersion);
    }
    return dependencyMinMuleVersionCompatible;
  }

  private boolean isDependencyMinMuleVersionCompatible(Dependency dependency, String runtimeVersion) {
    if (!MULE_PLUGIN_CLASSIFIER.equals(dependency.getClassifier())) {
      return true;
    }
    Optional<Artifact> dependencyArtifact = lookupArtifact(dependency);
    return dependencyArtifact
        .map(this::getPluginModel)
        .map(pluginModel -> isAtLeastMinMuleVersion(runtimeVersion, pluginModel.getMinMuleVersion())).orElse(true);
  }

  private MulePluginModel getPluginModel(Artifact artifact) {
    try {
      return JarUtils.loadFileContentFrom(artifact.getFile(), MULE_ARTIFACT_JSON_IN_JAR).map(jsonContent -> {
        MulePluginModel pluginModel = new MulePluginModelJsonSerializer().deserialize(new String(jsonContent));
        log.debug("Found minMuleVersion" + pluginModel.getMinMuleVersion() + " for plugin" + pluginModel.getName());
        return pluginModel;
      }).orElse(null);
    } catch (IOException e) {
      log.warn("An error occurred while reading " + artifact.getArtifactId() + " mule-artifact.json file", e);
      return null;
    }
  }

  private Optional<Artifact> lookupArtifact(Dependency dependency) {
    return project.getArtifacts().stream()
        .filter(artifact -> artifact.getGroupId().equals(dependency.getGroupId()))
        .filter(artifact -> artifact.getArtifactId().equals(dependency.getArtifactId())).findAny();
  }

  private boolean isNotBlacklisted(Dependency dependency) {
    return BLACKLISTED_DEPENDENCIES.stream().noneMatch(exclusion -> exclusion.excludes(dependency));
  }

  private static interface DependencyExclusion {

    boolean excludes(Dependency dependency);
  }

  private static class ArtifactExclusion implements DependencyExclusion {

    private String groupId;
    private String artifactId;

    ArtifactExclusion(String groupId, String artifactId) {
      this.groupId = groupId;
      this.artifactId = artifactId;
    }

    @Override
    public boolean excludes(Dependency dependency) {
      return groupId.equals(dependency.getGroupId()) && artifactId.equals(dependency.getArtifactId());
    }

  }

  private static class ClassifierExclusion implements DependencyExclusion {

    private String classifier;

    ClassifierExclusion(String classifier) {
      this.classifier = classifier;
    }

    @Override
    public boolean excludes(Dependency dependency) {
      return classifier.equals(dependency.getClassifier());
    }
  }

}
