/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.internal.generator;

import static java.util.Collections.emptyList;
import org.mule.maven.client.api.MavenReactorResolver;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;

import com.google.common.collect.ImmutableList;

import java.io.File;
import java.util.List;

import org.apache.maven.project.MavenProject;

/**
 * Retrieves extension's pom|jar file to resolve maven dependencies.
 *
 * @author Mulesoft Inc.
 * @since 1.1.0
 */
public class ExtensionMavenReactorResolver implements MavenReactorResolver {

  protected static final String MULE_PLUGIN_CLASSIFIER = "mule-plugin";
  private static final String POM = "pom";

  private final MavenProject project;

  public ExtensionMavenReactorResolver(MavenProject project) {
    this.project = project;
  }

  @Override
  public File findArtifact(BundleDescriptor bundleDescriptor) {
    if (isExtensionProject(bundleDescriptor)) {
      if (bundleDescriptor.getType().equals(POM)) {
        return project.getFile();
      } else {
        String buildDir = project.getBuild().getDirectory();
        return new File(buildDir + "/" + project.getArtifactId() + "-" + project.getVersion() + "-" + MULE_PLUGIN_CLASSIFIER
            + ".jar");
      }
    }
    return null;
  }

  @Override
  public List<String> findVersions(BundleDescriptor bundleDescriptor) {
    if (isExtensionProject(bundleDescriptor)) {
      return ImmutableList.of(this.project.getVersion());
    }
    return emptyList();
  }

  private boolean isExtensionProject(BundleDescriptor bundleDescriptor) {
    return this.project.getGroupId().equals(bundleDescriptor.getGroupId())
        && this.project.getArtifactId().equals(bundleDescriptor.getArtifactId())
        && this.project.getVersion().equals(bundleDescriptor.getVersion());
  }
}
