/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner.printer;


import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;
import org.mule.munit.plugin.maven.TargetRuntime;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.model.SuiteResult;
import org.mule.munit.plugin.maven.runner.model.TestResult;

import static java.lang.String.format;

/**
 * <p>
 * Prints the run result to a xml file following the surefire format.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class TextResultFilePrinter extends FileResultPrinter {

  private static final float MILLIS_FACTOR = 1000.0f;
  private static final String SUITE_RESULT_MESSAGE =
      "Tests run: %d - Failures: %d - Errors: %d - Skipped: %d - Time elapsed: %.3f sec";
  private static final String TEST_ERROR_MESSAGE = "ERROR - Test %s finished with an Error.\n";
  private static final String TEST_FAILED_MESSAGE = "FAILURE - Test %s finished with a Failure.\n";
  private static final String TEST_IGNORED_MESSAGE = "SKIPPED - Test %s was Skipped.\n";
  private static final String TEST_SUCCESS_MESSAGE = "SUCCESS - Test %s finished Successfully.\n";

  private static final String TXT_REPORT_HEADER = "munit";

  public TextResultFilePrinter(File reportBasePath, Log log) {
    super(reportBasePath, log);
  }

  public void print(TargetRuntime targetRuntime, RunResult runResult) {
    try {
      for (SuiteResult suiteResult : runResult.getSuites()) {
        printSuiteResult(targetRuntime, suiteResult);
      }
    } catch (FileNotFoundException | UnsupportedEncodingException e) {
      e.printStackTrace();
    }
  }

  public void printSuiteResult(TargetRuntime targetRuntime, SuiteResult result)
      throws FileNotFoundException, UnsupportedEncodingException {
    String suiteRelativePath = result.getSuitePath();
    String suiteReportName = suiteRelativePath.replace("/", ".");
    suiteReportName = suiteReportName.replace(".xml", ".txt");
    PrintStream out = getResultPrintStream(format("%s.%s.%s.%s", TXT_REPORT_HEADER, targetRuntime.getRuntimeProduct(),
                                                  targetRuntime.getRuntimeVersion(), suiteReportName));
    for (TestResult testResult : result.getTests()) {
      printTestResult(testResult, out);
    }
    out.println();
    if (result.hasError() || result.hasFailed()) {
      out.println(result.getCause());
    }
    String title = String.format(SUITE_RESULT_MESSAGE, result.getNumberOfTests(), result.getNumberOfFailures(),
                                 result.getNumberOfErrors(), result.getNumberOfIgnores(), result.getTime() / MILLIS_FACTOR);
    String titleFrame = StringUtils.repeat("=", title.length());
    out.println(titleFrame);
    out.println(title);
    out.println(titleFrame);
    out.flush();
  }

  public void printTestResult(TestResult testResult, PrintStream out) {
    if (testResult.hasError()) {
      out.format(TEST_ERROR_MESSAGE, testResult.getTestName());
      out.println(testResult.getCause());
    } else if (testResult.hasFailed()) {
      out.format(TEST_FAILED_MESSAGE, testResult.getTestName());
      out.println(testResult.getCause());
    } else if (testResult.isIgnored()) {
      out.format(TEST_IGNORED_MESSAGE, testResult.getTestName());
    } else {
      out.format(TEST_SUCCESS_MESSAGE, testResult.getTestName());
    }
    out.flush();
  }


}
