/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven;

import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static org.apache.commons.io.FileUtils.readFileToString;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.shared.utils.cli.CommandLineException;
import org.apache.maven.toolchain.ToolchainManager;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.repository.RemoteRepository;

import org.mule.munit.common.util.FileUtils;
import org.mule.munit.plugin.maven.locators.JVMLocator;
import org.mule.munit.plugin.maven.locators.RemoteRepositoriesLocator;
import org.mule.munit.plugin.maven.locators.RuntimeVersionsLocator;
import org.mule.munit.plugin.maven.locators.TestSuiteFilesLocator;
import org.mule.munit.plugin.maven.runner.JVMStarter;
import org.mule.munit.plugin.maven.runner.MessageHandlerFactory;
import org.mule.munit.plugin.maven.runner.consumer.ErrorStreamConsumer;
import org.mule.munit.plugin.maven.runner.consumer.RunnerStreamConsumer;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.printer.ResultPrinter;
import org.mule.munit.plugin.maven.runner.structure.WorkingDirectoryGenerator;
import org.mule.munit.plugin.maven.util.ArgLinesManager;
import org.mule.munit.plugin.maven.util.ClasspathManager;
import org.mule.munit.plugin.maven.util.JarFileFactory;
import org.mule.munit.plugin.maven.util.MuleApplicationModelLoader;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.plugin.maven.util.RuntimeVersionProviderFactory;
import org.mule.munit.plugin.maven.util.properties.UserPropertiesBuilder;
import org.mule.munit.remote.RemoteRunner;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.RunConfigurationParser;
import org.mule.munit.remote.api.project.ApplicationStructureGenerator;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

/**
 * Base Mojo to run MUnit tests
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public abstract class AbstractMunitMojo extends AbstractMojo {

  protected static final String ARG_TOKEN = "-";
  protected static final String RUN_CONFIGURATION_ARG = ARG_TOKEN + RunConfigurationParser.RUN_CONFIGURATION_PARAMETER;

  public static final String RUN_CONFIGURATION_JSON = "run-configuration.json";

  public static final String STARTER_CLASS_FILE = "munitstarter";
  public static final String MUNIT_PREVIOUS_RUN_PLACEHOLDER = "MUNIT_PREVIOUS_RUN_PLACEHOLDER";

  protected static final Class REMOTE_RUNNER_CLASS = RemoteRunner.class;
  private static final String SKIP_TESTS_PROPERTY = "skipTests";
  private static final String SKIP_MUNIT_TESTS_PROPERTY = "skipMunitTests";

  public static final String MULE_ARTIFACT_JSON_FILE_NAME = "mule-artifact.json";

  @Parameter(property = "munit.test")
  public String munitTest;

  @Parameter(property = "munit.tags")
  public String munitTags;

  @Parameter(property = "munit.debug")
  public String munitDebug;

  @Parameter(property = "runtimeVersion")
  public String runtimeVersion;

  @Parameter(property = "runtimeProduct")
  public String runtimeProduct;

  @Parameter(defaultValue = "${skipMunitTests}")
  public boolean skipMunitTests = false;

  @Parameter(defaultValue = "${project.build.directory}/test-mule/munit")
  public File munitTestsDirectory;

  @Parameter(property = "project", required = true)
  public MavenProject project;

  @Parameter(defaultValue = "${session}", readonly = true)
  public MavenSession session;

  @Parameter(defaultValue = "${repositorySystemSession}", readonly = true)
  public RepositorySystemSession repositorySystemSession;

  @Parameter(defaultValue = "${plugin.artifactId}")
  public String pluginArtifactId;

  @Parameter(property = "redirectTestOutputToFile", defaultValue = "false")
  public boolean redirectTestOutputToFile;

  @Parameter(defaultValue = "${project.build.directory}/munit-reports/output/")
  public File testOutputDirectory;

  @Parameter(property = "enableSurefireReports", defaultValue = "true")
  public boolean enableSurefireReports;

  @Parameter(defaultValue = "${project.build.directory}/surefire-reports/")
  public File surefireReportsFolder;

  @Parameter(property = "skipAfterFailure", defaultValue = "false")
  public boolean skipAfterFailure = false;

  @Parameter(property = "argLines")
  public List<String> argLines;

  @Parameter(property = "system.property.variables")
  public Map<String, String> systemPropertyVariables;

  @Parameter(property = "environment.variables")
  public Map<String, String> environmentVariables;

  @Parameter(property = "dynamic.ports")
  public List<String> dynamicPorts;

  @Parameter(property = "whitelistRepositories", defaultValue = "true")
  public boolean whitelistRepositories;

  @Component
  protected ToolchainManager toolchainManager;

  @Component
  public RepositorySystem repositorySystem;

  protected Gson gson = new GsonBuilder().setPrettyPrinting().create();

  public List<File> suiteFiles;
  public JVMLocator jvmLocator;
  public JarFileFactory jarFileFactory;
  public ClasspathManager classpathManager;
  public MessageHandlerFactory messageHandlerFactory;
  public Map<String, String> effectiveSystemProperties;
  public WorkingDirectoryGenerator workingDirectoryGenerator;
  public MuleApplicationModelLoader muleApplicationModelLoader;
  public RuntimeVersionProviderFactory runtimeVersionProviderFactory;
  public RuntimeVersionsLocator runtimeVersionsLocator;
  public ResultPrinterFactory resultPrinterFactory;
  private String starterJarFileName;

  protected void init() throws MojoExecutionException {
    this.jarFileFactory = new JarFileFactory();
    this.starterJarFileName = FileUtils.generateRandomFileName(STARTER_CLASS_FILE, ".jar");
    this.messageHandlerFactory = new MessageHandlerFactory(getLog());
    this.effectiveSystemProperties = getEffectiveSystemProperties();
    this.jvmLocator = new JVMLocator(session, toolchainManager, getLog());
    this.workingDirectoryGenerator = new WorkingDirectoryGenerator(getLog(), getApplicationStructureGenerator(), project);
    this.classpathManager = new ClasspathManager(REMOTE_RUNNER_CLASS, getLog());
    this.muleApplicationModelLoader = getMuleApplicationModelLoader();
    this.resultPrinterFactory = getResultPrinterFactory();
    this.runtimeVersionProviderFactory =
        new RuntimeVersionProviderFactory(repositorySystem, repositorySystemSession, locateRemoteRepositories(), getLog());
    this.runtimeVersionsLocator = new RuntimeVersionsLocator(runtimeVersionProviderFactory, getLog());
  }

  @Override
  public void execute() throws MojoExecutionException {
    if (!"true".equals(System.getProperty(SKIP_TESTS_PROPERTY))) {
      if (!skipMunitTests) {
        if (!hasExecutedBefore()) {
          if (hasSuites()) {
            init();
            doExecute();
          } else {
            getLog().warn("MUnit will not run, no MUnit suites found in your project");
          }
        } else {
          getLog().info("Skipping execution of munit because it has already been run");
        }
      } else {
        getLog().info(format("Run of %s skipped. Property [%s] was set to true", pluginArtifactId, SKIP_MUNIT_TESTS_PROPERTY));
      }
    } else {
      getLog().info(format("Run of %s skipped. Property [%s] was set to true", pluginArtifactId, SKIP_TESTS_PROPERTY));
    }
  }

  /**
   * This method avoids running MUnit more than once. This could happen because other phases such as the coverage-report, can
   * trigger the "test" phase since it needs its information.
   *
   * @return true if the MUnit run has already happened before
   */
  @SuppressWarnings("unchecked")
  protected boolean hasExecutedBefore() {
    Map<String, String> pluginContext = getPluginContext();
    if (pluginContext.containsKey(MUNIT_PREVIOUS_RUN_PLACEHOLDER)) {
      return true;
    }
    getPluginContext().put(MUNIT_PREVIOUS_RUN_PLACEHOLDER, MUNIT_PREVIOUS_RUN_PLACEHOLDER);
    return false;
  }

  public void doExecute() throws MojoExecutionException {

    Map<TargetRuntime, RunConfiguration> runConfigurations = getRunConfigurations();
    Map<TargetRuntime, RunResult> runResults = new HashMap<>();
    try {

      for (Map.Entry<TargetRuntime, RunConfiguration> runConfigurationEntry : runConfigurations.entrySet()) {

        workingDirectoryGenerator.generate();

        TargetRuntime targetRuntime = runConfigurationEntry.getKey();
        getLog().info(format("Running %s with version %s", targetRuntime.getRuntimeProduct(), targetRuntime.getRuntimeVersion()));

        RunConfiguration runConfiguration = runConfigurationEntry.getValue();

        if (runConfiguration.getSuitePaths().isEmpty()) {
          getLog().info("No MUnit test suite files found. No test will be run");
          return;
        }

        RunnerStreamConsumer streamConsumer = new RunnerStreamConsumer(messageHandlerFactory.create(redirectTestOutputToFile));
        ErrorStreamConsumer errorStreamConsumer = new ErrorStreamConsumer(redirectTestOutputToFile);
        int result = createTestRunExecutor(runConfiguration).execute(streamConsumer, errorStreamConsumer);

        RunResult runResult;
        if (result == 0) {
          runResult = streamConsumer.getRunResult();
          runResults.put(targetRuntime, runResult);
        } else {
          throw new MojoExecutionException("Build Failed", new MojoExecutionException(errorStreamConsumer.getOutput()));
        }

      }
      runResults.forEach(this::handleRunResult);
      failBuildIfNecessary(runResults);

    } catch (IOException | CommandLineException e) {
      throw new MojoExecutionException("Build Failed", e);
    }
  }

  protected JVMStarter createTestRunExecutor(RunConfiguration runConfiguration) throws IOException {
    File runConfigurationFile = saveRunConfigurationToFile(runConfiguration);
    File log4jConfigurationFile = new File(runConfiguration.getContainerConfiguration().getLog4JConfigurationFilePath());
    return createJVMStarter(runConfigurationFile, getEffectiveArgLines(starterJarFileName, log4jConfigurationFile));
  }

  protected void handleRunResult(TargetRuntime targetRuntime, RunResult runResult) {
    for (ResultPrinter printer : resultPrinterFactory.create()) {
      printer.print(targetRuntime, runResult);
    }
  }

  protected void failBuildIfNecessary(Map<TargetRuntime, RunResult> runResults) throws MojoExecutionException {
    if (runResults.values().stream().anyMatch(RunResult::hasFailed)) {
      throw new MojoExecutionException("MUnit Tests Failed");
    }
  }

  protected abstract Map<TargetRuntime, RunConfiguration> getRunConfigurations() throws MojoExecutionException;

  protected abstract ResultPrinterFactory getResultPrinterFactory();

  protected abstract ApplicationStructureGenerator getApplicationStructureGenerator() throws MojoExecutionException;

  protected abstract File getMuleApplicationJsonPath();

  public List<String> getEffectiveArgLines(String starterJarFileName, File log4jConfigurationFile) {
    return new ArgLinesManager(argLines, starterJarFileName, munitDebug, log4jConfigurationFile, getLog())
        .getEffectiveArgLines();
  }

  public JVMStarter createJVMStarter(File runConfigurationFile, List<String> argLines) throws IOException {
    getLog().debug("MUnit root folder found at: " + munitTestsDirectory.getAbsolutePath());

    File jarFile = jarFileFactory.create(classpathManager.getEffectiveClasspath(), REMOTE_RUNNER_CLASS.getCanonicalName(),
                                         new File(project.getBuild().getDirectory()), starterJarFileName);

    JVMStarter jvmStarter = new JVMStarter(getLog())
        .withJVM(jvmLocator.locate())
        .withWorkingDirectory(getWorkingDirectory())
        .withJar(jarFile)
        .withArgLines(argLines)
        .withSystemProperties(effectiveSystemProperties)
        .addEnvironmentVariables(environmentVariables);

    Map<String, File> fileArgLines = new HashMap<>();
    fileArgLines.put(RUN_CONFIGURATION_ARG, runConfigurationFile);
    jvmStarter.withArgLines(fileArgLines);

    return jvmStarter;
  }

  public File saveRunConfigurationToFile(RunConfiguration runConfiguration) throws IOException {
    File runConfigurationFile =
        Paths.get(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath(), RUN_CONFIGURATION_JSON)
            .toFile();
    saveAsJsonDataToFile(runConfiguration, runConfigurationFile);
    return runConfigurationFile;
  }

  protected void saveAsJsonDataToFile(Object data, File dataLocation) {
    try {
      if (dataLocation == null) {
        getLog().warn("Unable to save data, no destination file was provided");
        return;
      }
      dataLocation.getParentFile().mkdirs();
      dataLocation.createNewFile();
      FileUtils.writeStringToFile(dataLocation, gson.toJson(data), defaultCharset());
      getLog().debug("Data File saved in " + dataLocation);
    } catch (IOException e) {
      getLog().warn("Unable to save data to file:" + e.getMessage());
      getLog().debug(e);
    }
  }

  protected MuleApplicationModelLoader getMuleApplicationModelLoader() throws MojoExecutionException {
    return new MuleApplicationModelLoader(getMuleApplicationModel(), getLog()).withRuntimeProduct(runtimeProduct)
        .withRuntimeVersion(runtimeVersion);
  }

  protected MuleApplicationModel getMuleApplicationModel() throws MojoExecutionException {
    File muleApplicationJsonPath = getMuleApplicationJsonPath();
    try {
      return new MuleApplicationModelJsonSerializer().deserialize(readFileToString(muleApplicationJsonPath, defaultCharset()));
    } catch (IOException e) {
      String message = "Fail to read mule application file from " + muleApplicationJsonPath;
      getLog().error(message, e);
      throw new MojoExecutionException(message, e);
    }
  }

  public File getWorkingDirectory() {
    if (project != null) {
      return project.getBasedir();
    }
    return new File(".");
  }

  private Map<String, String> getEffectiveSystemProperties() {
    Map<String, String> effectiveSystemProperties = new HashMap<>();
    effectiveSystemProperties.putAll(getUserSystemProperties());
    // TODO MU-978
    effectiveSystemProperties.put("org.glassfish.grizzly.DEFAULT_MEMORY_MANAGER",
                                  "org.glassfish.grizzly.memory.HeapMemoryManager");
    return effectiveSystemProperties;
  }

  private Map<String, String> getUserSystemProperties() {
    return new UserPropertiesBuilder(project.getBuild().getDirectory(), getLog())
        .withSystemPropertyVariables(systemPropertyVariables)
        .withDynamicPorts(dynamicPorts)
        .withUserProperties(session != null ? session.getUserProperties() : null)
        .build();
  }

  private boolean hasSuites() {
    suiteFiles = new TestSuiteFilesLocator().locateFiles(munitTestsDirectory);
    return !suiteFiles.isEmpty();
  }

  private List<RemoteRepository> locateRemoteRepositories() {
    return new RemoteRepositoriesLocator(project).setWhiteListRepositories(whitelistRepositories).locate();
  }
}
