/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.locators;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.maven.project.MavenProject;
import org.eclipse.aether.repository.RemoteRepository;


/**
 * <p>
 * Locates all {@link RemoteRepository} for a {@link MavenProject}
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class RemoteRepositoriesLocator {

  private static final String SNAPSHOTS = "https://repository-master.mulesoft.org/nexus/content/repositories/snapshots";
  private static final String RELEASES = "https://repository-master.mulesoft.org/nexus/content/repositories/releases";
  private static final String RELEASES_EE = "https://repository-master.mulesoft.org/nexus/content/repositories/releases-ee";
  private static final String CI_SNAPSHOTS = "https://repository-master.mulesoft.org/nexus/content/repositories/ci-snapshots";
  private static final String SNAPSHOTS_PROXY = "https://repository.mulesoft.org/nexus/content/repositories/snapshots";
  private static final String RELEASES_PROXY = "https://repository.mulesoft.org/nexus/content/repositories/releases";
  private static final String RELEASES_EE_PROXY = "https://repository.mulesoft.org/nexus/content/repositories/releases-ee";
  private static final String CI_SNAPSHOTS_PROXY = "https://repository.mulesoft.org/nexus/content/repositories/ci-snapshots";
  private static final String RELEASES_SHORT_PROXY = "https://repository.mulesoft.org/releases";
  private static final String SNAPSHOTS_SHORT_PROXY = "https://repository.mulesoft.org/snapshots";

  protected static final List<String> WHITELISTED_REPOSITORIES =
      asList(RELEASES, SNAPSHOTS, RELEASES_PROXY, RELEASES_EE_PROXY, CI_SNAPSHOTS_PROXY, RELEASES_SHORT_PROXY,
             SNAPSHOTS_SHORT_PROXY, SNAPSHOTS_PROXY, RELEASES_EE, CI_SNAPSHOTS);

  private boolean whitelistRepositories = false;
  private List<RemoteRepository> remoteRepositories;

  public RemoteRepositoriesLocator(MavenProject mavenProject) {
    checkArgument(mavenProject != null, "Maven Project cannot be null");
    this.remoteRepositories = mavenProject.getRemoteProjectRepositories();
  }

  public RemoteRepositoriesLocator setWhiteListRepositories(boolean whitelistRepositories) {
    this.whitelistRepositories = whitelistRepositories;
    return this;
  }

  public List<RemoteRepository> locate() {
    return whitelistRepositories ? filterRepositories(remoteRepositories) : remoteRepositories;
  }

  private List<RemoteRepository> filterRepositories(List<RemoteRepository> repositories) {
    return repositories.stream().filter(this::isWhiteListed).collect(Collectors.toList());
  }

  private boolean isWhiteListed(RemoteRepository remoteRepository) {
    return WHITELISTED_REPOSITORIES.stream().anyMatch(whitelistedRepo -> remoteRepository.getUrl().startsWith(whitelistedRepo));
  }

}
