/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner.structure;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Optional.ofNullable;
import static org.mule.munit.remote.FolderNames.MUNIT_WORKING_DIR;

import java.io.File;
import java.nio.file.Path;
import java.util.Optional;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

import org.mule.munit.common.util.FileUtils;
import org.mule.munit.plugin.maven.util.MuleApplicationDependencyFinder;
import org.mule.munit.remote.api.project.ApplicationStructureGenerator;
import org.mule.munit.remote.api.project.DomainStructureGenerator;

/**
 * It knows how to generate a MUnit working directory and the folders that are required to run. It saves the sate, so after a
 * structure/folder has been created it won't do it again.
 * 
 * @author Mulesoft Inc.
 * @since 2.1.0
 */
public class WorkingDirectoryGenerator {

  protected Log log;
  private ApplicationStructureGenerator structureGenerator;
  private MavenProject project;
  private String dirTimestamp;
  private String destinationName;

  private File workingDirectory;
  private Path generatedDomainStructurePath;
  private Path generatedApplicationStructurePath;

  public WorkingDirectoryGenerator(Log log, ApplicationStructureGenerator applicationStructureGenerator, MavenProject project) {
    checkArgument(log != null, "Log must not be null");
    checkArgument(applicationStructureGenerator != null, "Application structure generator must not be null");
    checkArgument(project != null, "Project must not be null");

    this.log = log;
    this.project = project;
    this.structureGenerator = applicationStructureGenerator;
    this.dirTimestamp = Long.toString(System.nanoTime());
    this.destinationName = project.getArtifactId();
  }

  public void setDestinationName(String destinationName) {
    this.destinationName = destinationName;
  }

  /**
   * Generates all required folders for the MUnit working directory. Making sure previous runs in the same folder are erased
   * 
   * @return a file pointing to the new MUnit working directory
   */
  public Path generate() throws MojoExecutionException {
    deleteDirectories();
    Path path = generateWorkingDirectory();
    generateApplicationStructure();
    generateDomainStructure();
    return path;
  }

  /**
   * Provides the File to a new MUnit working directory that has been generated in the project's target and appended a timestamp.
   * It will create just one per instance.
   * 
   * @return a file pointing to the new MUnit working directory
   */
  public Path generateWorkingDirectory() {
    if (workingDirectory == null) {
      this.workingDirectory =
          new File(project.getBuild().getDirectory(), MUNIT_WORKING_DIR.value() + "-" + dirTimestamp);
    }
    return workingDirectory.toPath();
  }

  /**
   * It creates the application folder structure in the working directory
   *
   * @return the path to the generated structure
   * @throws MojoExecutionException
   */
  public Path generateApplicationStructure() throws MojoExecutionException {
    if (generatedApplicationStructurePath == null) {

      Path destinationPath = generateWorkingDirectory();

      try {
        log.debug("Attempting to create application structure destination: " + destinationPath);

        generatedApplicationStructurePath = structureGenerator.generate(destinationPath.resolve(destinationName));
      } catch (Exception e) {
        throw new MojoExecutionException("Fail to create application structure", e);
      }
    }
    return generatedApplicationStructurePath;
  }


  /**
   * It creates the domain folder structure in the working directory provided there is a mule-domain as dependency of the
   * application.
   * 
   * @return the path to the generated structure
   * @throws MojoExecutionException
   */
  public Optional<Path> generateDomainStructure() throws MojoExecutionException {
    if (generatedDomainStructurePath == null) {
      Optional<Artifact> domainArtifact = getMuleApplicationDependencyFinder().findMuleDomainArtifact();
      if (domainArtifact.isPresent()) {
        File domainFile = domainArtifact.get().getFile();
        try {
          log.debug("Attempting to create domain structure source: " + domainFile.getAbsolutePath()
              + " - destination: " + generateWorkingDirectory());

          generatedDomainStructurePath = getDomainStructureGenerator(domainArtifact, domainFile).generate();
        } catch (Exception e) {
          throw new MojoExecutionException("Fail to create domain structure", e);
        }
      }
    }
    return ofNullable(generatedDomainStructurePath);
  }

  /**
   * Clears all directories previously created
   */
  public void deleteDirectories() {
    ofNullable(workingDirectory).ifPresent(FileUtils::deleteQuietly);
    ofNullable(generatedDomainStructurePath).map(Path::toFile).ifPresent(FileUtils::deleteQuietly);
    ofNullable(generatedApplicationStructurePath).map(Path::toFile).ifPresent(FileUtils::deleteQuietly);
    this.workingDirectory = null;
    this.generatedDomainStructurePath = null;
    this.generatedApplicationStructurePath = null;
  }

  protected MuleApplicationDependencyFinder getMuleApplicationDependencyFinder() {
    return new MuleApplicationDependencyFinder(project);
  }

  protected DomainStructureGenerator getDomainStructureGenerator(Optional<Artifact> domainArtifact, File domainFile) {
    String domainName = new StringBuilder()
        .append(domainArtifact.get().getArtifactId()).append("-")
        .append(domainArtifact.get().getBaseVersion()).append("-")
        .append(domainArtifact.get().getClassifier()).toString();

    return new DomainStructureGenerator(domainFile.toPath(), generateWorkingDirectory(), domainName);
  }

}
