/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.locators;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toCollection;
import static org.mule.munit.plugin.maven.RuntimeProducts.CE;
import static org.mule.runtime.api.deployment.meta.Product.MULE;
import static org.mule.runtime.api.deployment.meta.Product.MULE_EE;

import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Stream;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;

import org.mule.munit.plugin.maven.ProductDiscoveryMode;
import org.mule.munit.plugin.maven.TargetRuntime;
import org.mule.munit.plugin.maven.util.RuntimeVersionProviderFactory;
import org.mule.runtime.api.deployment.meta.Product;
import org.mule.runtime.api.meta.MuleVersion;

import com.mulesoft.anypoint.discovery.api.RuntimeVersionProvider;
import com.mulesoft.anypoint.discovery.api.version.ArtifactVersion;


/**
 * <p>
 * Locates all runtime versions and obtains resulting {@link TargetRuntime} list
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class RuntimeVersionsLocator {

  private static final MuleVersion PATCH_VERSION_4_1 = new MuleVersion("4.1.1");

  private RuntimeVersionProviderFactory runtimeVersionProviderFactory;
  private ProductDiscoveryMode productDiscoveryMode;
  private String minMuleVersion;
  private boolean includeSnapshots;
  private Log log;

  public RuntimeVersionsLocator(RuntimeVersionProviderFactory runtimeVersionProviderFactory, Log log) {
    this.runtimeVersionProviderFactory = runtimeVersionProviderFactory;
    this.log = log;
  }

  public RuntimeVersionsLocator withMinMuleVersion(String minMuleVersion) {
    this.minMuleVersion = minMuleVersion;
    return this;
  }

  public RuntimeVersionsLocator withProductDiscoveryMode(ProductDiscoveryMode productDiscoveryMode) {
    this.productDiscoveryMode = productDiscoveryMode;
    return this;
  }

  public RuntimeVersionsLocator includingSnapshots(boolean includeSnapshots) {
    this.includeSnapshots = includeSnapshots;
    return this;
  }

  public Set<TargetRuntime> locate() throws MojoExecutionException {
    List<ArtifactVersion> eeVersions =
        productDiscoveryMode.supportsEe() ? locateRuntimes(runtimeVersionProviderFactory, MULE_EE) : emptyList();
    List<ArtifactVersion> ceVersions =
        productDiscoveryMode.supportsCe() ? locateRuntimes(runtimeVersionProviderFactory, MULE) : emptyList();

    Set<TargetRuntime> targetRuntimes = createTargetRuntimes(eeVersions, ceVersions)
        .filter(this::isNotCePatchVersion)
        .filter(this::includeSnapshots)
        .filter(this::greaterThanMinVersion)
        .collect(toCollection(TreeSet::new));

    log.debug("Discovered Mule runtimes: " + targetRuntimes);
    return targetRuntimes;
  }

  private boolean isNotCePatchVersion(TargetRuntime targetRuntime) {
    if (targetRuntime.getRuntimeProduct().equals(CE.value())) {
      MuleVersion muleVersion = new MuleVersion(targetRuntime.getRuntimeVersion());
      return muleVersion.getRevision() == 0 || muleVersion.equals(PATCH_VERSION_4_1);
    }
    return true;
  }

  private Stream<TargetRuntime> createTargetRuntimes(List<ArtifactVersion> eeVersions, List<ArtifactVersion> ceVersions) {
    Stream<TargetRuntime> ceRuntimes = ceVersions.stream().map(version -> toTargetRuntime(version, MULE));
    Stream<TargetRuntime> eeRuntimes = eeVersions.stream().map(version -> toTargetRuntime(version, MULE_EE));
    return Stream.concat(ceRuntimes, eeRuntimes);
  }

  private List<ArtifactVersion> locateRuntimes(RuntimeVersionProviderFactory versionProviderFactory, Product product)
      throws MojoExecutionException {
    return versionProviderFactory.create(product).map(RuntimeVersionProvider::all).orElse(emptyList());
  }

  private boolean greaterThanMinVersion(TargetRuntime targetRuntime) {
    return new MuleVersion(targetRuntime.getRuntimeVersion()).atLeast(minMuleVersion);
  }

  private boolean includeSnapshots(TargetRuntime targetRuntime) {
    MuleVersion muleVersion = new MuleVersion(targetRuntime.getRuntimeVersion());
    if (muleVersion.hasSuffix()) {
      return includeSnapshots && "SNAPSHOT".equals(muleVersion.getSuffix());
    }
    return true;
  }

  private TargetRuntime toTargetRuntime(ArtifactVersion artifactVersion, Product product) {
    return new TargetRuntime(artifactVersion.value(), product.name());
  }
}
