/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner.printer;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

import org.mule.munit.plugin.maven.TargetRuntime;
import org.mule.munit.plugin.maven.report.xml.CdataAwareXppDriver;
import org.mule.munit.plugin.maven.report.xml.Property;
import org.mule.munit.plugin.maven.report.xml.TestCase;
import org.mule.munit.plugin.maven.report.xml.TestSuite;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.model.SuiteResult;
import org.mule.munit.plugin.maven.runner.model.TestResult;

import com.thoughtworks.xstream.XStream;

/**
 * <p>
 * Prints the run result to a xml file following the surefire format.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class XmlResultPrinter extends FileResultPrinter {

  private static final float MILLIS_FACTOR = 1000.0f;
  private static final String XML_REPORT_HEADER = "TEST-munit";
  private static final String XML_HEADER = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + System.lineSeparator();

  private TestSuite suite;
  private String name;
  private Map<String, String> systemProperties;

  public XmlResultPrinter(File reportBasePath, Map<String, String> systemProperties, Log log) {
    super(reportBasePath, log);
    this.systemProperties = systemProperties;
  }

  public void print(TargetRuntime targetRuntime, RunResult runResult) {
    try {
      for (SuiteResult suiteResult : runResult.getSuites()) {
        printSuiteResult(targetRuntime, suiteResult);
      }
    } catch (FileNotFoundException | UnsupportedEncodingException e) {
      e.printStackTrace();
    }
  }

  public void printSuiteResult(TargetRuntime targetRuntime, SuiteResult result)
      throws FileNotFoundException, UnsupportedEncodingException {
    CdataAwareXppDriver xppDriver = new CdataAwareXppDriver();
    XStream xStream = new XStream(xppDriver);
    xStream.autodetectAnnotations(true);
    String suiteRelativePath = result.getSuitePath();
    suiteRelativePath += result.getParameterization().map(parameterization -> "[" + parameterization + "]").orElse(EMPTY);

    String suiteName = FilenameUtils.getName(suiteRelativePath).replace(".xml", EMPTY);
    this.name = String.format("%s.%s.%s", targetRuntime.getRuntimeProduct(), targetRuntime.getRuntimeVersion(), suiteName);
    this.suite = new TestSuite(dumpProperties(System.getProperties(), systemProperties), this.name);
    for (TestResult testResult : result.getTests()) {
      printTestResult(testResult);
    }
    suite.setErrors(result.getNumberOfErrors());
    suite.setFailures(result.getNumberOfFailures());
    suite.setTests(result.getNumberOfTests());
    suite.setTime(result.getTime() / MILLIS_FACTOR);
    suite.setSkipped(result.getNumberOfIgnores());

    String suiteReportName = this.name.replace("/", ".") + ".xml";
    PrintStream out = getResultPrintStream(format("%s.%s", XML_REPORT_HEADER, suiteReportName));
    out.print(XML_HEADER + xStream.toXML(suite));
  }

  public void printTestResult(TestResult testResult) {
    TestCase testCase = new TestCase(testResult.getTime() / MILLIS_FACTOR, name, testResult.getTestName());
    testCase.setSkipped(testResult.isIgnored());
    if (testResult.hasFailed()) {
      testCase.setFailure(testResult.getCause());
    }
    if (testResult.hasError()) {
      testCase.setError(testResult.getCause());
    }
    suite.add(testCase);
  }

  private List<Property> dumpProperties(Properties properties, Map<String, String> sysProps) {
    ArrayList<Property> testProperties = new ArrayList<>();
    for (Map.Entry<Object, Object> entry : properties.entrySet()) {
      if (!sysProps.containsKey(entry.getKey())) {
        testProperties.add(new Property((String) entry.getKey(), sanitize((String) entry.getValue())));
      }
    }
    for (Map.Entry<String, String> sysEntry : sysProps.entrySet()) {
      testProperties.add(new Property(sysEntry.getKey(), sysEntry.getValue()));
    }
    return testProperties;
  }

  private String sanitize(String value) {
    if (containsEscapesIllegalXml10(value)) {
      return escapeXml(value, true);
    } else {
      return value;
    }
  }

  private static boolean containsEscapesIllegalXml10(String message) {
    int size = message.length();
    for (int i = 0; i < size; i++) {
      if (isIllegalEscape(message.charAt(i))) {
        return true;
      }

    }
    return false;
  }

  private static boolean isIllegalEscape(char c) {
    return isIllegalEscape((int) c);
  }

  private static boolean isIllegalEscape(int c) {
    return c >= 0 && c < 32 && c != '\n' && c != '\r' && c != '\t';
  }

  private static String escapeXml(String text, boolean attribute) {
    StringBuilder sb = new StringBuilder(text.length() * 2);
    for (int i = 0; i < text.length(); i++) {
      char c = text.charAt(i);
      if (isIllegalEscape(c)) {
        sb.append(attribute ? "&#" : "&amp#").append((int) c).append(
                                                                     ';');
      } else {
        sb.append(c);
      }
    }
    return sb.toString();
  }
}
