/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.util;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Sets.newHashSet;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;
import org.mule.munit.plugin.maven.TargetRuntime;
import org.mule.munit.plugin.maven.locators.Log4J2ConfigurationLocator;
import org.mule.munit.plugin.maven.locators.TestSuiteFilesLocator;
import org.mule.munit.plugin.maven.runner.structure.WorkingDirectoryGenerator;
import org.mule.munit.remote.api.configuration.ContainerConfiguration.ContainerConfigurationBuilder;
import org.mule.munit.remote.api.configuration.MavenConfiguration;
import org.mule.munit.remote.api.configuration.NotifierConfiguration;
import org.mule.munit.remote.api.configuration.NotifierParameter;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.notifiers.ConsoleNotifier;

/**
 * Creates {@link RunConfiguration} based on basic MUnit parameters
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class BaseRunConfigurationFactory implements RunConfigurationFactory {

  public static final String RUN_TOKEN_CONSOLE_PARAMETER = "runToken";

  private Log log;
  private String munitTags;
  private String projectName;
  private Boolean skipAfterFailure;
  private TestSuiteFileFilter testSuiteFileFilter;
  private WorkingDirectoryGenerator workingDirectoryGenerator;
  private TargetRuntime targetRuntime;


  private File munitSrcFolder;
  private String runToken;

  private MavenProject project;
  private MavenSession session;

  public BaseRunConfigurationFactory(Log log, String projectName, String munitTest, String munitTags, Boolean skipAfterFailure,
                                     TargetRuntime targetRuntime, WorkingDirectoryGenerator workingDirectoryGenerator,
                                     File munitSrcFolder, MavenProject project, MavenSession session) {

    checkNotNull(log, "The log must not be null");
    checkNotNull(skipAfterFailure, "The skipAfterFailure must not be null");
    checkNotNull(targetRuntime, "The muleApplicationModelLoader must not be null nor empty");
    checkNotNull(workingDirectoryGenerator, "The WorkingDirectoryGenerator must not be null");
    checkNotNull(munitSrcFolder, "The munitSrcFolder must not be null");
    checkNotNull(project, "The project must not be null");
    checkNotNull(session, "The session must not be null");

    this.log = log;
    this.projectName = projectName;
    this.munitTags = munitTags;
    this.skipAfterFailure = skipAfterFailure;
    this.targetRuntime = targetRuntime;
    this.testSuiteFileFilter = new TestSuiteFileFilter(log, munitTest);

    this.munitSrcFolder = munitSrcFolder;
    this.runToken = UUID.randomUUID().toString();
    this.workingDirectoryGenerator = workingDirectoryGenerator;

    this.project = project;
    this.session = session;
  }

  public RunConfiguration create() throws MojoExecutionException {
    return getRunConfigurationBuilder().build();
  }

  protected MavenConfiguration.MavenConfigurationBuilder getMavenConfigurationBuilder() {
    DefaultSettingsSupplierFactory defaultSettingsSupplierFactory =
        new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());

    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = new MavenConfiguration.MavenConfigurationBuilder();
    mavenConfigurationBuilder.withMavenRepositoryDirectoryPath(session.getLocalRepository().getBasedir())
        .withSettingsXmlFilePath(session.getRequest().getUserSettingsFile().getAbsolutePath())
        .withGlobalSettingsXmlFilePath(session.getRequest().getGlobalSettingsFile().getAbsolutePath())
        .withForcePolicyUpdate(true)
        .withOfflineMode(session.isOffline())
        .withIgnoreArtifactDescriptorRepositories(false);
    defaultSettingsSupplierFactory.environmentSettingsSecuritySupplier()
        .ifPresent(file -> mavenConfigurationBuilder.withSecuritySettingsFilePath(file.getPath()));
    return mavenConfigurationBuilder;
  }

  protected RunConfiguration.RunConfigurationBuilder getRunConfigurationBuilder() throws MojoExecutionException {
    String testToRunName = testSuiteFileFilter.getTestNameRegEx();
    return new RunConfiguration.RunConfigurationBuilder().withRunToken(runToken)
        .withProjectName(projectName)
        .withTags(munitTags == null ? Collections.emptySet() : new HashSet<>(Arrays.asList(munitTags.split(","))))
        .withSkipAfterFailure(skipAfterFailure)
        .withTestNames(isBlank(testToRunName) ? Collections.emptySet() : newHashSet(testToRunName.split(",")))
        .withSuitePaths(locateMunitTestSuitesToRun())
        .withAllSuitePaths(locateAllMunitTestSuites())
        .withNotifierConfigurations(getNotifierConfigurations())
        .withContainerConfiguration(getContainerConfigurationBuilder().build())
        .withDomainLocation(workingDirectoryGenerator.generateDomainStructure()
            .map(domainPath -> domainPath.toFile().getAbsolutePath())
            .orElse(StringUtils.EMPTY));
  }

  protected List<NotifierConfiguration> getNotifierConfigurations() {
    List<NotifierParameter> consoleParameters = new ArrayList<>();
    consoleParameters.add(new NotifierParameter(RUN_TOKEN_CONSOLE_PARAMETER, String.class.getCanonicalName(), runToken));
    NotifierConfiguration consoleNotifierConfiguration = new NotifierConfiguration.NotifierConfigurationBuilder()
        .withClazz(ConsoleNotifier.class.getCanonicalName())
        .withParameters(consoleParameters)
        .build();

    List<NotifierConfiguration> notifierConfigurations = new ArrayList<>();
    notifierConfigurations.add(consoleNotifierConfiguration);
    return notifierConfigurations;
  }

  protected ContainerConfigurationBuilder getContainerConfigurationBuilder() {
    return new ContainerConfigurationBuilder()
        .withRuntimeId(targetRuntime.getRuntimeVersion())
        .withProduct(targetRuntime.getRuntimeProduct())
        .withMunitWorkingDirectoryPath(workingDirectoryGenerator.generateWorkingDirectory().toFile().getAbsolutePath())
        .withLog4JConfigurationFilePath(getLog4JConfigurationFilePath())
        .withMavenConfiguration(getMavenConfigurationBuilder().build());
  }

  protected Set<String> locateAllMunitTestSuites() {
    return new TestSuiteFilesLocator().locateFiles(munitSrcFolder).stream()
        .map(suiteFile -> munitSrcFolder.toURI().relativize(suiteFile.toURI()).toASCIIString())
        .collect(toSet());
  }

  protected Set<String> locateMunitTestSuitesToRun() {
    return locateAllMunitTestSuites().stream().filter(suitePath -> testSuiteFileFilter.shouldFilter(suitePath)).collect(toSet());
  }

  private String getLog4JConfigurationFilePath() {
    return getLog4JConfigurationFile().isPresent() ? getLog4JConfigurationFile().get().getAbsolutePath() : EMPTY;
  }

  private Optional<File> getLog4JConfigurationFile() {
    List<File> files = new Log4J2ConfigurationLocator(log).locateFiles(new File(project.getBuild().getDirectory()));
    return files.isEmpty() ? Optional.empty() : Optional.of(files.get(0));
  }


}
