/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.util;

import static java.lang.String.format;
import static java.util.Optional.of;
import static java.util.stream.Collectors.joining;

import java.util.List;
import java.util.Optional;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.repository.RemoteRepository;

import org.mule.munit.plugin.maven.locators.RemoteRepositoriesLocator;
import org.mule.runtime.api.deployment.meta.Product;

import com.mulesoft.anypoint.discovery.api.RuntimeVersionProvider;
import com.mulesoft.anypoint.discovery.api.aether.AetherRuntimeVersionProvider;
import com.mulesoft.anypoint.discovery.api.exception.NoLatestVersionFoundException;
import com.mulesoft.anypoint.discovery.api.exception.VersionDiscoveryException;

/**
 * <p>
 * Creates {@link RuntimeVersionProvider} based on the Mule runtime product
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RuntimeVersionProviderFactory {

  private Log log;
  private RepositorySystem repositorySystem;
  private RepositorySystemSession repositorySystemSession;
  private RemoteRepositoriesLocator remoteRepositoriesLocator;

  public RuntimeVersionProviderFactory(RepositorySystem repositorySystem, RepositorySystemSession repositorySystemSession,
                                       RemoteRepositoriesLocator remoteRepositoriesLocator, Log log) {
    this.repositorySystem = repositorySystem;
    this.repositorySystemSession = repositorySystemSession;
    this.remoteRepositoriesLocator = remoteRepositoriesLocator;
    this.log = log;
  }

  public Optional<RuntimeVersionProvider> create(Product product) throws MojoExecutionException {
    List<RemoteRepository> remoteRepositories = remoteRepositoriesLocator.locate();
    try {
      validateRepositories(remoteRepositories);
      return of(new AetherRuntimeVersionProvider(repositorySystem, repositorySystemSession, remoteRepositories, product));
    } catch (VersionDiscoveryException e) {
      throw new MojoExecutionException("An error occurred while discovering versions for product " + product
          + " in the following repositories " + remoteRepositories);
    } catch (NoLatestVersionFoundException e) {
      log.warn("Unable to find Mule Runtime versions for product " + product);
      return Optional.empty();
    }
  }

  private void validateRepositories(List<RemoteRepository> remoteRepositories) throws MojoExecutionException {
    if (remoteRepositories.isEmpty()) {
      throw new MojoExecutionException(format("%nNo repositories found to discover runtimes.%nFound:%n  %s%nbut needed one of:%n  %s%n",
                                              remoteRepositoriesLocator.locateAll().stream().map(RemoteRepository::getUrl)
                                                  .collect(joining(format(",%n  "))),
                                              String.join(format(",%n  "),
                                                          remoteRepositoriesLocator.getWhitelistedRepositories())));
    }
  }
}
