/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runtime;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.mule.runtime.api.meta.MuleVersion;

/**
 * Runtime version and product (CE or EE)
 *
 * @since 2.2.0
 * @author Mulesoft Inc.
 */
public class TargetRuntime implements Comparable<TargetRuntime> {

  static final String CE = "MULE";
  static final String EE = "MULE_EE";
  private static final List<String> PRODUCTS = Arrays.asList(CE, EE);

  private static final String TARGET_RUNTIME_SEPARATOR = ":";
  private MuleVersion runtimeVersion;
  private String runtimeProduct;

  public TargetRuntime(String runtimeVersion, String runtimeProduct) {
    checkArgument(isNotBlank(runtimeVersion), "Version cannot be blank");
    this.runtimeVersion = new MuleVersion(runtimeVersion);

    checkArgument(isNotBlank(runtimeProduct), "Product cannot be blank");
    checkArgument(PRODUCTS.contains(runtimeProduct), "Invalid product: %s", runtimeProduct);

    this.runtimeProduct = runtimeProduct;
  }

  public static TargetRuntime parse(String targetRuntime) {
    String[] versionAndProduct = StringUtils.split(targetRuntime, TARGET_RUNTIME_SEPARATOR);
    if (versionAndProduct.length != 2) {
      throw new IllegalArgumentException("Target runtime should be in the form of <product>:<version>. For example MULE_EE:4.1.1");
    }
    String product = versionAndProduct[0];
    String version = versionAndProduct[1];
    return new TargetRuntime(version, product);
  }

  public String getRuntimeProduct() {
    return runtimeProduct;
  }

  public String getRuntimeVersion() {
    return runtimeVersion.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    TargetRuntime that = (TargetRuntime) o;
    return this.runtimeVersion.equals(that.runtimeVersion) && this.runtimeProduct.equals(that.runtimeProduct);
  }

  @Override
  public int hashCode() {
    int result = runtimeVersion != null ? runtimeVersion.hashCode() : 0;
    result = 31 * result + (runtimeProduct != null ? runtimeProduct.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return runtimeProduct + TARGET_RUNTIME_SEPARATOR + runtimeVersion.toString();
  }

  @Override
  public int compareTo(TargetRuntime that) {
    if (that == null) {
      throw new NullPointerException("Can't compare against null");
    }

    if (this.equals(that)) {
      return 0;
    }

    if (this.runtimeProduct.equals(that.runtimeProduct)) {
      return this.runtimeVersion.newerThan(that.runtimeVersion) ? +1 : -1;
    } else {
      return this.runtimeProduct.equals(CE) ? -1 : +1;
    }
  }
}
