/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner.printer;


import static java.lang.Math.min;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

import org.mule.munit.plugin.maven.runtime.TargetRuntime;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.model.SuiteResult;
import org.mule.munit.plugin.maven.runner.model.TestResult;

/**
 * <p>
 * Prints the MUnit run result through the Maven Log
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MavenResultPrinter implements ResultPrinter {

  private static final String ERROR_TAG = "ERROR";
  private static final String FAILURE_TAG = "FAILED";
  private static final String SEPARATOR_TAG = " <<< ";
  private static final int MAX_RUN_STACKTRACE = 100;
  private static final int MAX_SUITE_STACKTRACE = 50;

  private Log log;

  public MavenResultPrinter(Log log) {
    this.log = log;
  }

  public void print(TargetRuntime runtime, RunResult runResult) {

    log.info("====================================================================================");
    log.info(format("MUnit Run Summary - Product: %s, Version: %s", runtime.getRuntimeProduct(), runtime.getRuntimeVersion()));
    log.info("====================================================================================");

    printRunStackTraceIfPresent(runResult);

    int testCount = runResult.getNumberOfTests();
    int errorCount = runResult.getNumberOfErrors();
    int failCount = runResult.getNumberOfFailures();
    int skipCount = runResult.getNumberOfIgnores();

    for (SuiteResult suiteResult : runResult.getSuites()) {
      List<TestResult> failingTests = suiteResult.getFailingTests();
      List<TestResult> errorTests = suiteResult.getErrorTests();

      printSuiteResult(suiteResult);
      printFailures(failingTests, log);
      printError(errorTests, log);
    }

    log.info("\t");
    log.info("====================================================================================");
    log.info(" > Tests:   \t" + testCount);
    log.info(" > Errors:  \t" + errorCount);
    log.info(" > Failures:\t" + failCount);
    log.info(" > Skipped: \t" + skipCount);
    log.info("====================================================================================");
  }

  private void printRunStackTraceIfPresent(RunResult runResult) {
    if (runResult.hasFailed() && isNotBlank(runResult.getStackTrace())) {
      String stackTrace = runResult.getStackTrace().substring(0, min(runResult.getStackTrace().length(), MAX_RUN_STACKTRACE));
      if (runResult.isRuntimeStartFailed()) {
        log.error("Run failed to start: " + stackTrace + "...");
      } else {
        log.error("An error occurred: " + stackTrace + "...");
      }
    }
  }

  private void printSuiteResult(SuiteResult suiteResult) {
    String unsuccessfulTag = StringUtils.EMPTY;
    if (suiteResult.isSuiteFailed() || suiteResult.isSuiteError() || suiteResult.isRuntimeStartFailed()) {
      unsuccessfulTag = SEPARATOR_TAG;
      unsuccessfulTag += suiteResult.isSuiteFailed() ? FAILURE_TAG : ERROR_TAG;
      String cause = suiteResult.getCause();
      if (isNotBlank(cause)) {
        unsuccessfulTag += " (" + cause.substring(0, min(cause.length(), MAX_SUITE_STACKTRACE)) + "...)";
      }
    }
    String suiteResultString =
        (suiteResult.getNumberOfTests() == 0 ? " (No tests run)" : getSuiteResultSummary(suiteResult)) + unsuccessfulTag;
    log.info(" >> " + suiteResult.getSuitePath() + getParameterization(suiteResult) + suiteResultString);
  }

  private void printFailures(List<TestResult> failingTests, Log log) {
    printUnsuccessfulTests(failingTests, FAILURE_TAG, log);
  }

  private void printError(List<TestResult> errorTests, Log log) {
    printUnsuccessfulTests(errorTests, ERROR_TAG, log);
  }

  private void printUnsuccessfulTests(List<TestResult> unsuccessfulTests, String unsuccessfulTag, Log log) {
    if (!unsuccessfulTests.isEmpty()) {
      for (TestResult result : unsuccessfulTests) {
        log.info("\t --- " + result.getTestName() + SEPARATOR_TAG + unsuccessfulTag);
      }
    }
  }

  private String getParameterization(SuiteResult suiteResult) {
    return suiteResult.getParameterization().map(parameterization -> "[" + parameterization + "]").orElse(EMPTY);
  }

  private String getSuiteResultSummary(SuiteResult suiteResult) {
    return " test result: " +
        "Tests: " + suiteResult.getNumberOfTests() +
        ", Errors: " + suiteResult.getNumberOfErrors() +
        ", Failures: " + suiteResult.getNumberOfFailures() +
        ", Skipped: " + suiteResult.getNumberOfIgnores();
  }
}
