/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner.consumer;

import static org.mule.munit.common.protocol.message.MessageField.FIELD_TOKEN;
import static org.mule.munit.common.protocol.message.MessageField.NEW_LINE_ESCAPE_TOKEN;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.shared.utils.cli.StreamConsumer;

import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.message.RunMessageParser;
import org.mule.munit.common.protocol.message.TestStatus;
import org.mule.munit.plugin.maven.runner.handler.RunnerMessageHandler;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.model.SuiteResult;
import org.mule.munit.plugin.maven.util.SuiteResultUtils;
import org.mule.munit.remote.api.configuration.RunConfiguration;

import java.util.List;

/**
 * <p>
 * A Stream Consumer that reads the Standard Output and identifies the Test Run protocol
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunnerStreamConsumer implements StreamConsumer {

  private RunResult runResult = new RunResult();
  private RunnerMessageHandler runnerMessageHandler;
  private RunMessageParser runnerParser;
  private RunConfiguration runConfiguration;

  public RunnerStreamConsumer(RunnerMessageHandler runnerMessageHandler) {
    this.runnerMessageHandler = runnerMessageHandler;
    this.runnerParser = new RunMessageParser(new RunStreamConsumerEventListener());
  }

  public void consumeLine(String s) {
    if (s.startsWith(FIELD_TOKEN)) {
      handleProtocolLine(s.replace(FIELD_TOKEN, StringUtils.EMPTY).replaceAll(NEW_LINE_ESCAPE_TOKEN, System.lineSeparator()));
    } else {
      runnerMessageHandler.handleStandardOutputLine(s);
    }
  }

  public void setRunConfiguration(RunConfiguration runConfiguration) {
    this.runConfiguration = runConfiguration;
  }

  protected void handleProtocolLine(String message) {
    runnerParser.parseAndNotify(message);
  }

  private class RunStreamConsumerEventListener implements RunEventListener {

    @Override
    public void notifyRunFinish() {
      runResult.setRunFinished(true);
    }

    @Override
    public void notifyCoverageReport(String coverageReportJson) {
      runResult.setCoverageReport(coverageReportJson);
    }

    @Override
    public void notifySuiteStart(String suite, String parameterization, int numberOfTests) {
      runResult.testSuiteStarted(suite, parameterization);
      runnerMessageHandler.handleSuiteStart(suite, parameterization);
    }

    @Override
    public void notifyContainerFailure(String stackTrace) {
      runResult.containerStartFailed(stackTrace);
      runnerMessageHandler.handleSuiteStartFailure(stackTrace);
      if (runResult.getSuites().isEmpty()) {
        createErrorSuite();
      }
    }

    @Override
    public void notifyContainerFailure(String suite, String parameterization, String stackTrace) {
      runResult.testSuiteStartFailed(suite, parameterization, stackTrace);
      runnerMessageHandler.handleSuiteStartFailure(suite, parameterization, stackTrace);
      if (runResult.getSuite(suite, parameterization) == null) {
        createErrorSuite();
      }
    }

    @Override
    public void notifyBeforeSuiteEnd(String name, String stackTrace, TestStatus status) {
      notifyBeforeOrAfterSuite(stackTrace, status);
    }

    @Override
    public void notifyTestStart(String name, String description, boolean isIgnored) {
      runResult.testStarted(name);
    }

    @Override
    public void notifyTestEnd(String name, String stackTrace, TestStatus status, long elapsedTime) {
      switch (status) {
        case SUCCESS:
          runResult.testFinished(name, elapsedTime);
          break;
        case FAILURE:
          runResult.testFailure(name, stackTrace, elapsedTime);
          break;
        case IGNORED:
          runResult.testIgnored(name, elapsedTime);
          break;
        case ERROR:
          runResult.testError(name, stackTrace, elapsedTime);
          break;
      }
    }

    @Override
    public void notifyAfterSuiteEnd(String name, String stackTrace, TestStatus status) {
      notifyBeforeOrAfterSuite(stackTrace, status);
    }

    @Override
    public void notifySuiteEnd(String suite, String parameterization, long elapsedTime) {
      runResult.suiteFinished(elapsedTime);
      runnerMessageHandler.handleSuiteFinished(runResult.getSuite(suite, parameterization));
    }

    @Override
    public void notifyUnexpectedError(String stackTrace) {
      runResult.unexpectedError(stackTrace);
      runnerMessageHandler.handleUnexpectedError(stackTrace);
      if (runResult.getSuites().isEmpty()) {
        createErrorSuite();
      }
    }

    @Override
    public void notifySuiteUnexpectedError(String name, String stackTrace) {
      runResult.testSuiteUnexpectedError(name, stackTrace);
      runnerMessageHandler.handleUnexpectedError(stackTrace);
      boolean suiteExists = runResult.getSuites().stream().anyMatch(suite -> suite.getSuitePath().equals(name));
      if (!suiteExists) {
        createErrorSuite();
      }
    }

  }

  public RunResult getRunResult() {
    runnerMessageHandler.handleRunResult(runResult);
    return runResult;
  }

  private void createErrorSuite() {
    List<SuiteResult> suiteResults = SuiteResultUtils.createErrorSuites(runResult, runConfiguration.getSuitePaths());
    suiteResults.forEach(suite -> runResult.addSuite(suite));
  }

  private void notifyBeforeOrAfterSuite(String stackTrace, TestStatus status) {
    switch (status) {
      case ERROR:
        runResult.suiteError(stackTrace);
        runnerMessageHandler.handleSuiteError(runResult.getCurrentSuite());
        break;
      case FAILURE:
        runResult.suiteFailure(stackTrace);
        runnerMessageHandler.handleSuiteFailure(runResult.getCurrentSuite());
        break;
    }
  }

}
