/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven;

import static org.mule.munit.plugin.maven.runtime.DiscoverProduct.DISCOVER_PRODUCT;
import static org.mule.munit.plugin.maven.runtime.RuntimeProducts.EE;
import static org.mule.munit.remote.api.project.MuleApplicationStructureGenerator.RUN_CONFIGURATION_JSON;

import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.io.FileUtils.readFileToString;

import org.mule.munit.common.util.FileUtils;
import org.mule.munit.plugin.maven.locators.JVMLocator;
import org.mule.munit.plugin.maven.locators.ProductVersionsLocator;
import org.mule.munit.plugin.maven.locators.RemoteRepositoriesLocator;
import org.mule.munit.plugin.maven.locators.TestSuiteFilesLocator;
import org.mule.munit.plugin.maven.runner.JVMStarter;
import org.mule.munit.plugin.maven.runner.MessageHandlerFactory;
import org.mule.munit.plugin.maven.runner.consumer.ErrorStreamConsumer;
import org.mule.munit.plugin.maven.runner.consumer.RunnerStreamConsumer;
import org.mule.munit.plugin.maven.runner.model.Debugger;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.printer.ResultPrinter;
import org.mule.munit.plugin.maven.runner.structure.WorkingDirectoryGenerator;
import org.mule.munit.plugin.maven.runtime.DiscoverProduct;
import org.mule.munit.plugin.maven.runtime.Product;
import org.mule.munit.plugin.maven.runtime.ProductConfiguration;
import org.mule.munit.plugin.maven.runtime.RuntimeConfiguration;
import org.mule.munit.plugin.maven.runtime.TargetProduct;
import org.mule.munit.plugin.maven.util.ArgLinesManager;
import org.mule.munit.plugin.maven.util.ClasspathManager;
import org.mule.munit.plugin.maven.util.JarFileFactory;
import org.mule.munit.plugin.maven.util.MuleApplicationModelLoader;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.plugin.maven.util.RuntimeVersionProviderFactory;
import org.mule.munit.plugin.maven.util.properties.UserPropertiesBuilder;
import org.mule.munit.remote.RemoteRunner;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.RunConfigurationParser;
import org.mule.munit.remote.api.project.ApplicationStructureGenerator;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.tools.model.Deployment;
import org.mule.tools.model.anypoint.AnypointDeployment;
import org.mule.tools.model.anypoint.CloudHubDeployment;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.ProjectBuilder;
import org.apache.maven.shared.utils.cli.CommandLineException;
import org.apache.maven.toolchain.ToolchainManager;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;

/**
 * Base Mojo to run MUnit tests
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public abstract class AbstractMunitMojo extends AbstractMojo {

  protected static final String ARG_TOKEN = "-";
  protected static final String RUN_CONFIGURATION_ARG = ARG_TOKEN + RunConfigurationParser.RUN_CONFIGURATION_PARAMETER;

  public static final String STARTER_CLASS_FILE = "munitstarter";
  public static final String MUNIT_PREVIOUS_RUN_PLACEHOLDER = "MUNIT_PREVIOUS_RUN_PLACEHOLDER";

  protected static final Class REMOTE_RUNNER_CLASS = RemoteRunner.class;
  private static final String SKIP_TESTS_PROPERTY = "skipTests";
  private static final String SKIP_MUNIT_TESTS_PROPERTY = "skipMunitTests";

  public static final String MULE_ARTIFACT_JSON_FILE_NAME = "mule-artifact.json";

  @Parameter(property = "munit.test")
  public String munitTest;

  @Parameter(property = "munit.tags")
  public String munitTags;

  @Parameter(property = "munit.failIfNoTests", defaultValue = "true")
  public boolean munitFailIfNoTests = true;

  @Parameter(property = "munit.debug", defaultValue = "false")
  public String munitDebug;

  @Parameter(property = "runtimeVersion")
  public String runtimeVersion;

  @Parameter(property = "runtimeProduct")
  public String runtimeProduct;

  @Parameter(defaultValue = "${skipMunitTests}")
  public boolean skipMunitTests = false;

  @Parameter(defaultValue = "false")
  public boolean clearParameters = false;

  @Parameter(defaultValue = "${project.build.directory}/test-mule/munit")
  public File munitTestsDirectory;

  @Parameter(defaultValue = "${project.build.directory}/test-mule/munit", readonly = true)
  public File defaultMunitTestsDirectory;

  @Parameter(property = "project", required = true)
  public MavenProject project;

  @Parameter(defaultValue = "${session}", readonly = true)
  public MavenSession session;

  @Parameter(defaultValue = "${repositorySystemSession}", readonly = true)
  public RepositorySystemSession repositorySystemSession;

  @Parameter(defaultValue = "${plugin.artifactId}")
  public String pluginArtifactId;

  @Parameter(property = "redirectTestOutputToFile", defaultValue = "false")
  public boolean redirectTestOutputToFile;

  @Parameter(defaultValue = "${project.build.directory}/munit-reports/output/")
  public File testOutputDirectory;

  @Parameter(property = "enableSurefireReports", defaultValue = "true")
  public boolean enableSurefireReports;

  @Parameter(property = "enableSonarReports", defaultValue = "true")
  public boolean enableSonarReports;

  @Parameter(defaultValue = "${project.build.directory}/surefire-reports/")
  public File surefireReportsFolder;

  @Parameter(defaultValue = "${project.build.directory}/sonar-reports/")
  public File sonarReportsFolder;

  @Parameter(property = "skipAfterFailure", defaultValue = "false")
  public boolean skipAfterFailure = false;

  @Parameter(property = "munit.jvm")
  public String jvm;

  @Parameter(property = "argLines")
  public List<String> argLines;

  @Parameter(property = "system.property.variables")
  public Map<String, String> systemPropertyVariables;

  @Parameter(property = "environment.variables")
  public Map<String, String> environmentVariables;

  @Parameter(property = "dynamic.ports")
  public List<String> dynamicPorts;

  /**
   * Set this to true to ignore a failure during testing. Its use is NOT RECOMMENDED, but quite convenient on occasion.
   */
  @Parameter(property = "maven.test.failure.ignore", defaultValue = "false")
  public boolean testFailureIgnore;

  /**
   * @deprecated since 2.4 use {@link #productConfiguration} instead.
   */
  @Parameter
  @Deprecated
  public RuntimeConfiguration runtimeConfiguration;

  @Parameter
  public ProductConfiguration productConfiguration;

  @Component
  protected ToolchainManager toolchainManager;

  @Component
  public RepositorySystem repositorySystem;

  @Component
  protected ProjectBuilder projectBuilder;

  @Component
  protected org.apache.maven.repository.RepositorySystem mavenRepositorySystem;

  @Parameter(property = "project.build.directory", required = true)
  public File outputDirectory;

  @Parameter(readonly = true, required = true, defaultValue = "${localRepository}")
  public ArtifactRepository localRepository;

  @Parameter(readonly = true, required = true, defaultValue = "${project.remoteArtifactRepositories}")
  public List<ArtifactRepository> remoteArtifactRepositories;

  @Parameter(defaultValue = "${project.basedir}")
  public File projectBaseFolder;

  @Parameter
  public CloudHubDeployment cloudHubDeployment;

  protected Gson gson = new GsonBuilder().setPrettyPrinting().create();

  public List<File> suiteFiles;
  public JVMLocator jvmLocator;
  public JarFileFactory jarFileFactory;
  public ClasspathManager classpathManager;
  public MessageHandlerFactory messageHandlerFactory;
  public Map<String, String> effectiveSystemProperties;
  public WorkingDirectoryGenerator workingDirectoryGenerator;
  public MuleApplicationModelLoader muleApplicationModelLoader;
  public RuntimeVersionProviderFactory runtimeVersionProviderFactory;
  public ProductVersionsLocator productVersionsLocator;
  public ResultPrinterFactory resultPrinterFactory;
  private String starterJarFileName = "";
  protected Debugger debugger;

  protected void init() throws MojoExecutionException {
    this.jarFileFactory = new JarFileFactory();
    this.starterJarFileName = FileUtils.generateRandomFileName(STARTER_CLASS_FILE, ".jar");
    this.messageHandlerFactory = new MessageHandlerFactory(getLog());
    this.validateProperties();
    this.effectiveSystemProperties = getEffectiveSystemProperties();
    this.jvmLocator = new JVMLocator(session, jvm, toolchainManager, getLog());
    this.workingDirectoryGenerator = new WorkingDirectoryGenerator(getLog(), getApplicationStructureGenerator(), project);
    this.classpathManager = new ClasspathManager(REMOTE_RUNNER_CLASS, getLog());
    this.muleApplicationModelLoader = getMuleApplicationModelLoader();
    this.resultPrinterFactory = getResultPrinterFactory();

    initConfiguration();

    this.runtimeVersionProviderFactory =
        new RuntimeVersionProviderFactory(repositorySystem, repositorySystemSession, createRemoteRepositoriesLocator(), getLog());
    this.productVersionsLocator = new ProductVersionsLocator(runtimeVersionProviderFactory, getLog());
    this.debugger = Debugger.fromString(munitDebug);
  }

  protected void initConfiguration() {
    if (this.runtimeConfiguration != null) {
      // Keep backwards compatibility
      this.productConfiguration = ProductConfiguration.builder()
          .from(productConfiguration)
          .withRuntimeConfiguration(RuntimeConfiguration.builder()
              .from(runtimeConfiguration)
              .build())
          .withOverrides(session.getUserProperties())
          .build();
    } else {
      this.productConfiguration = ProductConfiguration.builder()
          .from(productConfiguration)
          .withOverrides(session.getUserProperties())
          .build();
    }
    this.runtimeConfiguration = null;
  }

  @Override
  public void execute() throws MojoExecutionException {
    if (!"true".equals(System.getProperty(SKIP_TESTS_PROPERTY))) {
      if (!skipMunitTests) {
        if (!hasExecutedBefore()) {
          if (hasSuites()) {
            init();
            doExecute();
          } else {
            getLog().warn("MUnit will not run, no MUnit suites found in your project");
          }
        } else {
          getLog().info("Skipping execution of munit because it has already been run");
        }
      } else {
        getLog().info(format("Run of %s skipped. Property [%s] was set to true", pluginArtifactId, SKIP_MUNIT_TESTS_PROPERTY));
      }
    } else {
      getLog().info(format("Run of %s skipped. Property [%s] was set to true", pluginArtifactId, SKIP_TESTS_PROPERTY));
    }
  }

  /**
   * This method avoids running MUnit more than once. This could happen because other phases such as the coverage-report, can
   * trigger the "test" phase since it needs its information.
   *
   * @return true if the MUnit run has already happened before
   */
  @SuppressWarnings("unchecked")
  protected boolean hasExecutedBefore() {
    Map<String, String> pluginContext = getPluginContext();
    if (pluginContext.containsKey(MUNIT_PREVIOUS_RUN_PLACEHOLDER)) {
      return true;
    }
    getPluginContext().put(MUNIT_PREVIOUS_RUN_PLACEHOLDER, MUNIT_PREVIOUS_RUN_PLACEHOLDER);
    return false;
  }

  public void doExecute() throws MojoExecutionException {

    Map<TargetProduct, RunConfiguration> runConfigurations = getRunConfigurations();
    Map<TargetProduct, RunResult> runResults = new TreeMap<>();
    try {

      for (Map.Entry<TargetProduct, RunConfiguration> runConfigurationEntry : runConfigurations.entrySet()) {

        RunConfiguration runConfiguration = runConfigurationEntry.getValue();
        workingDirectoryGenerator.generate(runConfiguration);

        TargetProduct targetProduct = runConfigurationEntry.getKey();
        getLog().info(format("Running %s with version %s", targetProduct.getProduct(), targetProduct.getVersion()));

        if (runConfiguration.getSuitePaths().isEmpty()) {
          if (munitFailIfNoTests) {
            throw new MojoExecutionException("No tests suites were found! (Set -Dmunit.failIfNoTests=false to ignore this error.)");
          } else {
            getLog().info("No MUnit test suite files found. No test will be run");
            return;
          }
        }

        RunnerStreamConsumer streamConsumer = new RunnerStreamConsumer(messageHandlerFactory.create(redirectTestOutputToFile));
        streamConsumer.setRunConfiguration(runConfiguration);
        ErrorStreamConsumer errorStreamConsumer = new ErrorStreamConsumer(redirectTestOutputToFile);
        int result = createTestRunExecutor(runConfiguration).execute(streamConsumer, errorStreamConsumer);

        RunResult runResult;
        if (result == 0) {
          runResult = streamConsumer.getRunResult();
          runResults.put(targetProduct, runResult);
        } else {
          throw new MojoExecutionException("Build Failed", new MojoExecutionException(errorStreamConsumer.getOutput()));
        }

      }
      runResults.forEach(this::handleRunResult);
      failBuildIfNecessary(runResults);

    } catch (IOException | CommandLineException e) {
      throw new MojoExecutionException("Build Failed", e);
    }
  }

  protected JVMStarter createTestRunExecutor(RunConfiguration runConfiguration) throws IOException {
    File runConfigurationFile = saveRunConfigurationToFile(runConfiguration);
    File log4jConfigurationFile = new File(runConfiguration.getContainerConfiguration().getLog4JConfigurationFilePath());
    return createJVMStarter(runConfigurationFile, getEffectiveArgLines(starterJarFileName, log4jConfigurationFile));
  }

  protected void handleRunResult(TargetProduct targetProduct, RunResult runResult) {
    for (ResultPrinter printer : resultPrinterFactory.create()) {
      printer.print(targetProduct, runResult);
    }
  }

  protected void failBuildIfNecessary(Map<TargetProduct, RunResult> runResults) throws MojoExecutionException {
    if (runResults.values().stream().anyMatch(RunResult::finishedWithErrors)) {
      throw new MojoExecutionException("There was an error running MUnit tests");
    }

    if (!testFailureIgnore && runResults.values().stream().anyMatch(RunResult::hasFailed)) {
      throw new MojoExecutionException("MUnit Tests Failed");
    }

    if (munitFailIfNoTests && runResults.values().stream().allMatch(result -> result.getNumberOfTests() == 0)) {
      throw new MojoExecutionException("No tests were executed! (Set -Dmunit.failIfNoTests=false to ignore this error.)");
    }
  }

  protected abstract ResultPrinterFactory getResultPrinterFactory();

  protected abstract ApplicationStructureGenerator getApplicationStructureGenerator() throws MojoExecutionException;

  protected abstract File getMuleApplicationJsonPath();

  protected abstract RunConfiguration createRunConfiguration(TargetProduct targetProduct) throws MojoExecutionException;

  protected Map<TargetProduct, RunConfiguration> getRunConfigurations() throws MojoExecutionException {
    Map<TargetProduct, RunConfiguration> runConfigurations = new TreeMap<>();
    List<TargetProduct> effectiveTargetRuntimes = new ArrayList<>();

    if (shouldRunSingleRuntime()) {
      addDefaultTargetProducts(effectiveTargetRuntimes);
    } else {
      List<String> additionalProducts = productConfiguration.getAdditionalProducts();

      List<DiscoverProduct> productsToDiscover = productConfiguration.getDiscoverProducts()
          .stream()
          .filter(discoverProduct -> cloudHubDeployment == null && !discoverProduct.isSkipped())
          .collect(toList());

      if (additionalProducts != null && !additionalProducts.isEmpty()) {
        effectiveTargetRuntimes.addAll(additionalProducts.stream().map(TargetProduct::parse).collect(toList()));
      }
      if (!productsToDiscover.isEmpty()) {
        for (DiscoverProduct discoverProduct : productsToDiscover) {
          effectiveTargetRuntimes.addAll(discoverProducts(muleApplicationModelLoader.getRuntimeVersion(), discoverProduct));
        }
      } else {
        addDefaultTargetProducts(effectiveTargetRuntimes);
      }
    }

    for (TargetProduct targetRuntime : effectiveTargetRuntimes) {
      RunConfiguration runConfiguration = createRunConfiguration(targetRuntime);
      runConfigurations.put(targetRuntime, runConfiguration);
    }
    return runConfigurations;
  }

  private boolean shouldRunSingleRuntime() {
    return runtimeVersion != null || runtimeProduct != null;
  }

  private Set<TargetProduct> discoverProducts(String version, DiscoverProduct discoverProduct)
      throws MojoExecutionException {
    Product product = discoverProduct.getProductId()
        .orElseThrow(() -> new MojoExecutionException(format("Invalid Product Configuration: the auto discovery of products was "
            + "enabled but no product kind was specified, neither in the plugin nor using the command line argument '%s'. One of "
            + "the following products has to be selected: %s",
                                                             DISCOVER_PRODUCT,
                                                             asList(Product.values()))));

    if (muleApplicationModelLoader.getRuntimeProduct().equals(EE.value()) && !product.supportsEe()) {
      throw new MojoExecutionException(format("Product is EE only but was configured to discover %s runtimes.",
                                              product.name()));
    }

    productVersionsLocator
        .includingSnapshots(discoverProduct.isIncludeSnapshots())
        .withMinVersion(discoverProduct.getMinVersion().orElse(version))
        .withProduct(product)
        .usingLatestPatches(discoverProduct.isUseLatestPatches());

    return productVersionsLocator.locate();
  }

  private void addDefaultTargetProducts(List<TargetProduct> effectiveTargetProducts) {
    effectiveTargetProducts
        .add(new TargetProduct(muleApplicationModelLoader.getRuntimeVersion(),
                               muleApplicationModelLoader.getRuntimeProduct().equals("MULE")
                                   ? Product.MULE_CE
                                   : Product.MULE_EE));
  }

  public List<String> getEffectiveArgLines(String starterJarFileName, File log4jConfigurationFile) {
    if (!Optional.ofNullable(debugger).map(Debugger::getAddDebugger).orElse(false)) {
      return new ArgLinesManager(argLines, starterJarFileName, munitDebug, log4jConfigurationFile, getLog())
          .getEffectiveArgLines();
    }

    List<String> effectiveArgLines =
        new ArgLinesManager(argLines, starterJarFileName, null, log4jConfigurationFile, getLog()).getEffectiveArgLines();
    return Stream
        .concat(effectiveArgLines.stream(), debugger.getArguments().stream())
        .filter(StringUtils::isNotBlank)
        .collect(Collectors.toList());
  }

  public JVMStarter createJVMStarter(File runConfigurationFile, List<String> argLines) throws IOException {
    getLog().debug("MUnit root folder found at: " + munitTestsDirectory.getAbsolutePath());

    File jarFile = jarFileFactory.create(classpathManager.getEffectiveClasspath(), REMOTE_RUNNER_CLASS.getCanonicalName(),
                                         new File(project.getBuild().getDirectory()), starterJarFileName);

    JVMStarter jvmStarter = new JVMStarter(getLog())
        .withJVM(jvmLocator.locate())
        .withWorkingDirectory(getWorkingDirectory())
        .withJar(jarFile)
        .withArgLines(argLines)
        .withSystemProperties(effectiveSystemProperties)
        .addEnvironmentVariables(environmentVariables);

    Map<String, File> fileArgLines = new HashMap<>();
    fileArgLines.put(RUN_CONFIGURATION_ARG, runConfigurationFile);

    jvmStarter.withArgLines(fileArgLines);
    return jvmStarter;
  }

  public File saveRunConfigurationToFile(RunConfiguration runConfiguration) throws IOException {
    File runConfigurationFile =
        Paths.get(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath(), RUN_CONFIGURATION_JSON)
            .toFile();
    saveAsJsonDataToFile(runConfiguration, runConfigurationFile);
    return runConfigurationFile;
  }

  protected void saveAsJsonDataToFile(Object data, File dataLocation) {
    try {
      if (dataLocation == null) {
        getLog().warn("Unable to save data, no destination file was provided");
        return;
      }
      dataLocation.getParentFile().mkdirs();
      dataLocation.createNewFile();
      FileUtils.writeStringToFile(dataLocation, gson.toJson(data), defaultCharset());
      getLog().debug("Data File saved in " + dataLocation);
    } catch (IOException e) {
      getLog().warn("Unable to save data to file:" + e.getMessage());
      getLog().debug(e);
    }
  }

  protected MuleApplicationModelLoader getMuleApplicationModelLoader() throws MojoExecutionException {
    return new MuleApplicationModelLoader(getMuleApplicationModel(), getLog()).withRuntimeProduct(runtimeProduct)
        .withRuntimeVersion(runtimeVersion);
  }

  protected MuleApplicationModel getMuleApplicationModel() throws MojoExecutionException {
    File muleApplicationJsonPath = getMuleApplicationJsonPath();
    try {
      return new MuleApplicationModelJsonSerializer().deserialize(readFileToString(muleApplicationJsonPath, defaultCharset()));
    } catch (IOException e) {
      String message = "Fail to read mule application file from " + muleApplicationJsonPath;
      getLog().error(message, e);
      throw new MojoExecutionException(message, e);
    }
  }

  public File getWorkingDirectory() {
    if (project != null) {
      return project.getBasedir();
    }
    return new File(".");
  }

  private Map<String, String> getEffectiveSystemProperties() {
    Map<String, String> effectiveSystemProperties = new HashMap<>();
    effectiveSystemProperties.putAll(getUserSystemProperties());
    // TODO MU-978
    effectiveSystemProperties.put("org.glassfish.grizzly.DEFAULT_MEMORY_MANAGER",
                                  "org.glassfish.grizzly.memory.HeapMemoryManager");
    return effectiveSystemProperties;
  }

  private Map<String, String> getUserSystemProperties() {
    UserPropertiesBuilder builder = new UserPropertiesBuilder(project.getBuild().getDirectory(), getLog())
        .withSystemPropertyVariables(systemPropertyVariables)
        .withDynamicPorts(dynamicPorts)
        .withUserProperties(session != null ? session.getUserProperties() : null);

    Deployment deployment = getDeployments().stream().filter(Objects::nonNull).findFirst().orElse(null);

    if (deployment != null && deployment instanceof AnypointDeployment) {
      AnypointDeployment anypointDeployment = (AnypointDeployment) deployment;
      builder.withUserCredentials(anypointDeployment.getUsername(), anypointDeployment.getPassword());
    }

    return builder.build();
  }

  private boolean hasSuites() {
    suiteFiles = new TestSuiteFilesLocator().locateFiles(munitTestsDirectory);
    return !suiteFiles.isEmpty();
  }

  protected RemoteRepositoriesLocator createRemoteRepositoriesLocator() {
    return new RemoteRepositoriesLocator(project);
  }

  public List<Deployment> getDeployments() {
    return Arrays.asList(cloudHubDeployment);
  }

  public void validateProperties() throws MojoExecutionException {

    if (!FileUtils.isInDirectory(defaultMunitTestsDirectory, munitTestsDirectory)) {
      throw new MojoExecutionException("munitTestsDirectory property must be a subfolder of ${project.build.directory}/test-mule/munit.");
    }

    if (cloudHubDeployment == null) {
      return;
    }

    if (dynamicPorts != null && !dynamicPorts.isEmpty()) {
      throw new MojoExecutionException("DynamicPorts defined in the pom file do not work on CloudHub. Use munit:dynamic-port instead.");
    }

    if (environmentVariables != null && !environmentVariables.isEmpty()) {
      getLog()
          .warn("Environment variables will work as a placeholder properties when run on CloudHub. No OS environment properties will be set.");
    }
  }
}
