/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner.model;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.ToIntFunction;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;

import com.google.gson.Gson;

/**
 * <p>
 * A model that represents the result of the whole MUnit run
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunResult {

  private SuiteResult currentSuite = null;
  private String stackTrace = StringUtils.EMPTY;

  private ApplicationCoverageReport coverageReport;
  private Map<String, List<String>> suiteOutputs = Collections.emptyMap();

  private boolean runFinished = false;
  private boolean coverageEnabled = true;
  private boolean unexpectedError = false;

  private boolean runtimeStartFailed = false;

  private Map<String, SuiteResult> suites = new HashMap<>();

  public void containerStartFailed(String stackTrace) {
    runtimeStartFailed = true;
    this.stackTrace = stackTrace;
  }

  public void testSuiteStartFailed(String suitePath, String parameterization, String stackTrace) {
    SuiteResult suite = new SuiteResult();
    suite.setSuitePath(suitePath);
    suite.setParameterization(parameterization);
    suite.setRuntimeStartFailed(true);
    suite.setCause(stackTrace);
    suites.put(getSuiteId(suitePath, parameterization), suite);
  }

  public void testSuiteStarted(String suiteFullPath, String parameterization) {
    SuiteResult suite = new SuiteResult();
    suite.setSuitePath(suiteFullPath);
    suite.setParameterization(parameterization);
    suites.put(getSuiteId(suiteFullPath, parameterization), suite);
    currentSuite = suite;
  }

  public void testStarted(String testName) {
    currentSuite.add(testName, new TestResult(testName));
  }

  public void testFailure(String testName, String stackTrace, long elapsedTime) {
    TestResult test = getOrCreateTest(testName);
    test.setFailed(true);
    test.setCause(stackTrace);
    test.setTime(elapsedTime);
  }

  public void testError(String testName, String stackTrace, long elapsedTime) {
    TestResult test = getOrCreateTest(testName);
    test.setError(true);
    test.setCause(stackTrace);
    test.setTime(elapsedTime);
  }

  public void testFinished(String testName, long elapsedTime) {
    TestResult test = getOrCreateTest(testName);
    test.setFinished(true);
    test.setTime(elapsedTime);
  }

  public void testIgnored(String testName, long elapsedTime) {
    TestResult test = getOrCreateTest(testName);
    test.setIgnored(true);
    test.setTime(elapsedTime);
  }

  public void addSuite(SuiteResult result) {
    suites.put(result.getSuitePath(), result);
  }

  public void suiteFinished(long elapsedTime) {
    currentSuite.setSuiteFinished(true);
    currentSuite.setTime(elapsedTime);
    currentSuite = null;
  }

  public void suiteFailure(String stackTrace) {
    currentSuite.setSuiteFailed(true);
    currentSuite.setCause(stackTrace);
  }

  public void suiteError(String stackTrace) {
    currentSuite.setSuiteError(true);
    currentSuite.setCause(stackTrace);
  }

  public void testSuiteUnexpectedError(String suiteFullPath, String stackTrace) {
    SuiteResult suite = new SuiteResult();
    suite.setSuitePath(suiteFullPath);
    suite.setSuiteError(true);
    suite.setCause(stackTrace);
    suites.put(suiteFullPath, suite);

    currentSuite = null;
  }

  public void unexpectedError(String stackTrace) {
    currentSuite = null;
    unexpectedError = true;
    runFinished = true;
    this.stackTrace = stackTrace;
  }

  public Collection<SuiteResult> getSuites() {
    return suites.values();
  }

  public boolean isRunFinished() {
    return runFinished;
  }

  public void setRunFinished(boolean runFinished) {
    this.runFinished = runFinished;
  }

  public SuiteResult getSuite(String suitePath, String parameterization) {
    return suites.get(getSuiteId(suitePath, parameterization));
  }

  public int getNumberOfTests() {
    return summarize(SuiteResult::getNumberOfTests);
  }

  public int getNumberOfIgnores() {
    return summarize(SuiteResult::getNumberOfIgnores);
  }

  public int getNumberOfErrors() {
    return summarize(SuiteResult::getNumberOfErrors);
  }

  public int getNumberOfFailures() {
    return summarize(SuiteResult::getNumberOfFailures);
  }

  private int summarize(ToIntFunction<SuiteResult> predicate) {
    return getSuites().stream().mapToInt(predicate).sum();
  }

  public SuiteResult getCurrentSuite() {
    return currentSuite;
  }

  public Optional<ApplicationCoverageReport> getApplicationCoverageReport() {
    return Optional.ofNullable(coverageReport);
  }

  public void setCoverageReport(ApplicationCoverageReport report) {
    this.coverageReport = report;
  }

  public void setCoverageReport(String jsonReport) {
    this.coverageReport = new Gson().fromJson(jsonReport, ApplicationCoverageReport.class);
  }

  public Map<String, List<String>> getSuiteOutputs() {
    return suiteOutputs;
  }

  public void setSuiteOutputs(Map<String, List<String>> suiteOutputs) {
    this.suiteOutputs = suiteOutputs;
  }

  public boolean hasFailed() {
    return finishedWithErrors() || suites.values().stream().anyMatch(suite -> (suite.hasFailed() || suite.hasError()));
  }

  public boolean finishedWithErrors() {
    return unexpectedError || runtimeStartFailed;
  }

  public void setCoverageEnabled(boolean runCoverage) {
    this.coverageEnabled = runCoverage;
  }

  public boolean isCoverageEnabled() {
    return coverageEnabled;
  }

  public String getStackTrace() {
    return stackTrace;
  }

  public boolean isRuntimeStartFailed() {
    return runtimeStartFailed;
  }

  private String getSuiteId(String suitePath, String parameterization) {
    return suitePath + "|" + parameterization;
  }

  private TestResult getOrCreateTest(String testName) {
    TestResult test = currentSuite.getTest(testName);
    if (test == null) {
      TestResult testResult = new TestResult(testName);
      currentSuite.add(testName, testResult);
      return testResult;
    }
    return test;
  }

}
