/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.util;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;
import org.mule.munit.remote.api.configuration.RunConfiguration;

import java.util.ArrayList;
import java.util.List;

import static org.mule.munit.common.util.Preconditions.checkArgument;
import static org.mule.munit.common.util.Preconditions.checkNotNull;

/**
 * The class handles the filtering logic for the MUnit test suite files
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class TestSuiteFileFilter {

  private static final String SINGLE_TEST_NAME_TOKEN = "#";
  private static final String TEST_DELIMITER = ",";

  private final Log log;

  private List<String> testSuiteRegEx = new ArrayList<String>();

  private List<RunConfiguration.Test> testNameRegEx = new ArrayList<>();

  public TestSuiteFileFilter(Log log, String filter) {
    checkNotNull(log, "The log must not be null");
    this.log = log;
    if (StringUtils.isNotBlank(filter)
        && StringUtils.countMatches(filter, SINGLE_TEST_NAME_TOKEN) == 1) {
      // this block is here to keep backwards compatibility
      String suite = filter.substring(0, filter.indexOf(SINGLE_TEST_NAME_TOKEN));
      testSuiteRegEx.add(suite);
      for (String test : filter.substring(filter.indexOf(SINGLE_TEST_NAME_TOKEN) + 1).split(TEST_DELIMITER)) {
        testNameRegEx.add(RunConfiguration.Test.of(suite, test));
      }
    } else if (StringUtils.isNotBlank(filter)
        && StringUtils.countMatches(filter, SINGLE_TEST_NAME_TOKEN) > 1) {

      for (String test : filter.split(TEST_DELIMITER)) {
        testSuiteRegEx.add(test.substring(0, test.indexOf(SINGLE_TEST_NAME_TOKEN)));
        testNameRegEx.add(RunConfiguration.Test.of(test.substring(0, test.indexOf(SINGLE_TEST_NAME_TOKEN)),
                                                   test.substring(test.indexOf(SINGLE_TEST_NAME_TOKEN) + 1)));
      }
    } else {
      testSuiteRegEx.add(filter);
    }
  }

  public List<RunConfiguration.Test> getTestNameRegEx() {
    return testNameRegEx;
  }

  /**
   * Evaluates if the provided file name matches the pattern.
   *
   * @param fileName
   * @return true the fileName matches the filter regular expression or if there is no regular expression filter
   */
  public boolean shouldFilter(String fileName) {
    checkArgument(StringUtils.isNotBlank(fileName), "The fileName should not be null nor empty");
    if (testSuiteRegEx.stream().anyMatch(suite -> StringUtils.isNotBlank(suite))) {
      boolean result = testSuiteRegEx.stream().anyMatch(suite -> fileName.matches(suite));
      if (!result) {
        log.debug("MUnit Test Suite file " + fileName + " skipped. It doesn't match filter criteria: [" + testSuiteRegEx + "]");
        return false;
      }
    }
    return true;
  }



}
