/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.fips;

import static java.io.File.createTempFile;
import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Collections.emptyList;
import static org.apache.commons.io.FileUtils.writeStringToFile;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.model.Dependency;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.ArtifactRequest;
import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.eclipse.aether.resolution.ArtifactResult;


/**
 * Aids with the resolution of any additional arguments for the runner in the case of FIPS testing.
 * @author Mulesoft Inc.
 */
public class FipsArgumentsResolver {

  private static final String JAVA_SECURITY_PROPERTIES_FILE_PROPERTY = "java.security.properties";
  private static final String FIPS_SECURITY_PROPERTIES =
      "security.provider.1=org.bouncycastle.jcajce.provider.BouncyCastleFipsProvider\n"
          + "security.provider.2=org.bouncycastle.jsse.provider.BouncyCastleJsseProvider fips:BCFIPS\n"
          + "security.provider.3=sun.security.provider.Sun\n"
          + "security.provider.4=sun.security.rsa.SunRsaSign\n"
          + "security.provider.5=sun.security.ec.SunEC\n"
          + "security.provider.6=com.sun.net.ssl.internal.ssl.Provider BCFIPS\n"
          + "security.provider.7=com.sun.crypto.provider.SunJCE\n"
          + "security.provider.8=sun.security.jgss.SunProvider\n"
          + "security.provider.9=com.sun.security.sasl.Provider\n"
          + "security.provider.10=org.jcp.xml.dsig.internal.dom.XMLDSigRI\n"
          + "security.provider.11=sun.security.smartcardio.SunPCSC\n"
          + "security.provider.12=apple.security.AppleProvider\n"
          + "ssl.KeyManagerFactory.algorithm=PKIX\n"
          + "ssl.TrustManagerFactory.algorithm=PKIX\n"
          + "keystore.type=PKCS12";

  private final Log log;
  private final RepositorySystem repositorySystem;
  private final RepositorySystemSession repositorySystemSession;
  private final List<RemoteRepository> remoteRepositories;
  private final File fipsSecurityPropertiesFile;
  private final List<Dependency> additionalFipsDependencies;

  // Cached resolution of additional FIPS dependencies (not thread-safe)
  private List<String> resolvedClassPathEntriesForAdditionalFipsDependencies;

  public FipsArgumentsResolver(Log log,
                               RepositorySystem repositorySystem,
                               RepositorySystemSession repositorySystemSession,
                               List<RemoteRepository> remoteRepositories,
                               List<Dependency> additionalFipsDependencies)
      throws IOException {
    this.log = log;
    this.repositorySystem = repositorySystem;
    this.repositorySystemSession = repositorySystemSession;
    this.remoteRepositories = remoteRepositories;
    this.additionalFipsDependencies = additionalFipsDependencies;

    fipsSecurityPropertiesFile = createSecurePropertiesFile();
  }

  /**
   * Resolves the artifacts indicated as additional FIPS dependencies and returns their locations as a list that can be
   * added to the classpath.
   *
   * @return The classpath entries with the artifacts indicated as additional FIPS dependencies.
   * @throws MojoExecutionException If there is any error resolving the dependencies.
   */
  public List<String> getClassPathEntriesForFipsProviders() throws MojoExecutionException {
    if (resolvedClassPathEntriesForAdditionalFipsDependencies == null) {
      resolvedClassPathEntriesForAdditionalFipsDependencies = doGetClassPathEntriesForFipsProviders();
    }

    return resolvedClassPathEntriesForAdditionalFipsDependencies;
  }

  /**
   * Builds the secure properties file override to be used as a JVM argument.
   */
  public String getSecurePropertiesArgument() {
    return "-D" + JAVA_SECURITY_PROPERTIES_FILE_PROPERTY + "=" + fipsSecurityPropertiesFile.getAbsolutePath();
  }

  private List<String> doGetClassPathEntriesForFipsProviders() throws MojoExecutionException {
    if (additionalFipsDependencies == null) {
      return emptyList();
    }

    List<String> list = new ArrayList<>();
    for (Dependency additionalFipsDependency : additionalFipsDependencies) {
      String s = resolveDependencyToPath(additionalFipsDependency);
      list.add(s);
    }
    return list;
  }

  private File createSecurePropertiesFile() throws IOException {
    File fipsSecurityPropertiesFile = createTempFile("tmp", "fips.security.properties");
    fipsSecurityPropertiesFile.deleteOnExit();
    writeStringToFile(fipsSecurityPropertiesFile, FIPS_SECURITY_PROPERTIES, defaultCharset());
    return fipsSecurityPropertiesFile;
  }

  private String resolveDependencyToPath(Dependency dependency) throws MojoExecutionException {
    org.eclipse.aether.artifact.Artifact artifact;
    try {
      artifact = new DefaultArtifact(dependency.getGroupId(),
                                     dependency.getArtifactId(),
                                     dependency.getClassifier(),
                                     dependency.getType(),
                                     dependency.getVersion());
    } catch (IllegalArgumentException e) {
      throw new MojoExecutionException(e.getMessage(), e);
    }

    ArtifactRequest request = new ArtifactRequest(artifact, remoteRepositories, null);

    log.info(format("Resolving artifact %s from %s", artifact, remoteRepositories));

    ArtifactResult result;
    try {
      result = repositorySystem.resolveArtifact(repositorySystemSession, request);
    } catch (ArtifactResolutionException e) {
      throw new MojoExecutionException(e.getMessage(), e);
    }

    log.info(format("Resolved artifact %s to %s from %s", artifact, result.getArtifact().getFile(), result.getRepository()));

    return result.getArtifact().getFile().getPath();
  }
}
