/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.util;

import static org.mule.munit.plugin.maven.AbstractMunitMojo.MULE_ARTIFACT_JSON_FILE_NAME;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.apache.commons.io.IOUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.meta.MuleVersion;

import org.apache.maven.plugin.logging.Log;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

/**
 * Reads the mule application model and gives access to its properties based on the user settings
 * 
 * @since 2.0.0
 * @author Mulesoft Inc.
 */
public class MuleApplicationModelLoader {

  private static final Pattern VERSION_PATTERN = Pattern.compile("support-tools-installer-\\d.\\d.\\d(-(SNAPSHOT|\\d+))?");
  public static final MuleVersion MIN_MULE_GA_VERSION = new MuleVersion("4.1.1");

  private final MuleApplicationModel muleApplicationModel;
  private String runtimeVersion;
  private String runtimeProduct;
  private String runtimeLocalDistribution;
  private final Log log;

  public MuleApplicationModelLoader(MuleApplicationModel muleApplicationModel, Log log) {
    this.muleApplicationModel = muleApplicationModel;
    this.log = log;
  }

  public MuleApplicationModelLoader withRuntimeVersion(String runtimeVersion) {
    this.runtimeVersion = runtimeVersion;
    return this;
  }

  public MuleApplicationModelLoader withRuntimeProduct(String runtimeProduct) {
    this.runtimeProduct = runtimeProduct;
    return this;
  }

  public MuleApplicationModelLoader withRuntimeLocalDistribution(String runtimeLocalDistribution) {
    this.runtimeLocalDistribution = runtimeLocalDistribution;
    return this;
  }

  public String getRuntimeVersion() throws MojoExecutionException, IOException {
    if (isNotBlank(runtimeLocalDistribution)) {
      return getVersionFromLocalDistribution(runtimeLocalDistribution);
    }

    if (isNotBlank(runtimeVersion)) {
      return runtimeVersion;
    }
    String runtimeVersion = getMinMuleVersion();
    log.debug("Runtime version set to " + runtimeVersion + " obtained from the " + MULE_ARTIFACT_JSON_FILE_NAME + " file");
    return runtimeVersion;
  }

  public String getRuntimeProduct() {
    if (isNotBlank(runtimeProduct)) {
      return runtimeProduct;
    }
    String runtimeProduct = muleApplicationModel.getRequiredProduct().name();
    log.debug("Runtime product set to " + runtimeProduct + " obtained from the " + MULE_ARTIFACT_JSON_FILE_NAME + " file");
    return runtimeProduct;
  }

  private String getMinMuleVersion() {
    MuleVersion minMuleVersion = new MuleVersion(muleApplicationModel.getMinMuleVersion());
    if (minMuleVersion.atLeast(MIN_MULE_GA_VERSION)) {
      return muleApplicationModel.getMinMuleVersion();
    }
    return MIN_MULE_GA_VERSION.toCompleteNumericVersion();
  }

  private String getVersionFromLocalDistribution(String path) throws MojoExecutionException, IOException {
    String message = "'" + path + "' is not a valid standalone mule distribution";

    File file = new File(path);
    if (!file.exists() || file.isFile()) {
      throw new MojoExecutionException(message);
    }

    File version = new File(file, "version.txt");
    if (version.exists() && version.isFile()) {
      return IOUtils.toString(new FileReader(version));
    }

    File tools = new File(file, "tools");
    if (!tools.exists() || tools.isFile()) {
      throw new MojoExecutionException(message);
    }

    return Stream.of(Objects.requireNonNull(tools.list()))
        .map(fileName -> {
          Matcher matcher = VERSION_PATTERN.matcher(fileName);
          if (!matcher.find()) {
            return null;
          }

          return matcher.group(0).substring(24);
        })
        .filter(Objects::nonNull)
        .findFirst()
        .orElseThrow(() -> new MojoExecutionException(message));

  }
}
