/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.project;

import static org.mule.munit.common.util.Preconditions.checkArgument;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.apache.commons.lang3.StringUtils;

/**
 * It create a exploded domain structure
 *
 * @author Mulesoft Inc.
 * @since 2.1.0
 */
public class DomainStructureGenerator {

  private Boolean exploded;
  private Path domainJarFilePath;
  private Path destinationFolderPath;

  public DomainStructureGenerator(Path domainJarFilePath, Path destinationFolder, String domainName) {
    this(domainJarFilePath, destinationFolder, domainName, false);
  }

  public DomainStructureGenerator(Path domainJarFilePath, Path destinationFolder, String domainName, Boolean exploded) {
    checkArgument(domainJarFilePath != null, "The domain jar file must not be null");
    checkArgument(destinationFolder != null, "The destination folder must not be null");
    checkArgument(StringUtils.isNotBlank(domainName), "The application name must not be null nor empty");

    this.exploded = exploded;
    this.domainJarFilePath = domainJarFilePath;
    this.destinationFolderPath = destinationFolder.resolve(domainName);
  }

  /**
   * Generates an exploded test package
   * 
   * @return the path to the generated structure
   * @throws Exception in case of failure
   */
  public Path generate() throws Exception {
    destinationFolderPath.toFile().mkdirs();
    if (exploded) {
      return createExplodedPackage();
    } else {
      return createJarPackaged();
    }
  }

  private Path createJarPackaged() throws Exception {
    return Files.copy(domainJarFilePath, destinationFolderPath.resolve(domainJarFilePath.getFileName()));
  }

  private Path createExplodedPackage() throws Exception {
    JarFile domainJarFile = new JarFile(domainJarFilePath.toFile());

    Enumeration enumEntries = domainJarFile.entries();
    while (enumEntries.hasMoreElements()) {
      JarEntry file = (JarEntry) enumEntries.nextElement();
      File f = new File(destinationFolderPath.toFile().getCanonicalPath() + File.separator + file.getName());
      if (file.isDirectory()) {
        f.mkdir();
        continue;
      }

      InputStream is = domainJarFile.getInputStream(file);
      FileOutputStream fos = new FileOutputStream(f);
      while (is.available() > 0) {
        fos.write(is.read());
      }
      fos.close();
      is.close();
    }

    domainJarFile.close();

    return destinationFolderPath;
  }

}
