/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.project;

import static org.mule.munit.common.util.Preconditions.checkArgument;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Optional.empty;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.codehaus.plexus.archiver.Archiver.DUPLICATES_ADD;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.tools.api.packager.structure.FolderNames.CLASSES;
import static org.mule.tools.api.packager.structure.FolderNames.MAVEN;
import static org.mule.tools.api.packager.structure.FolderNames.MULE_SRC;
import static org.mule.tools.api.packager.structure.FolderNames.MUNIT;
import static org.mule.tools.api.packager.structure.FolderNames.REPOSITORY;
import static org.mule.tools.api.packager.structure.FolderNames.TEST_CLASSES;
import static org.mule.tools.api.packager.structure.FolderNames.TEST_MULE;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.runtime.utils.MuleVersion;
import org.mule.tools.api.packager.Pom;
import org.mule.tools.api.packager.archiver.MuleArchiver;
import org.mule.tools.api.packager.archiver.MuleExplodedArchiver;
import org.mule.tools.api.packager.builder.MulePackageBuilder;
import org.mule.tools.api.packager.packaging.PackagingOptions;
import org.mule.tools.api.packager.sources.DefaultValuesMuleArtifactJsonGenerator;
import org.mule.tools.api.packager.sources.MuleArtifactContentResolver;
import org.mule.tools.api.packager.structure.FolderNames;
import org.mule.tools.api.packager.structure.ProjectStructure;
import org.mule.tools.api.util.MavenComponents;
import org.mule.tools.api.util.SourcesProcessor;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;

import org.apache.commons.io.FileUtils;


/**
 * It create a exploded application structure based on a source folder containing all the required folders and files.
 *
 * The exploded application contains the test resources.
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
public class MuleApplicationStructureGenerator implements ApplicationStructureGenerator {

  public static final String RUN_CONFIGURATION_JSON = "run-configuration.json";
  public static final Path RUN_CONFIGURATION_PATH = Paths.get(META_INF.value(), MUNIT.value(), RUN_CONFIGURATION_JSON);
  private static final Gson gson = new GsonBuilder().setPrettyPrinting().create();
  private static final MuleVersion LOCAL_REPOSITORY_MIN_MULE_VERSION = new MuleVersion("4.2.2");

  protected Path baseFolderPath;
  protected Path originFolderPath;
  protected MulePackageBuilder packageBuilder;
  protected Pom pom;
  protected List<BundleDependency> dependencies = Collections.emptyList();
  protected MavenComponents mavenComponents;
  protected boolean heavyWeight;


  public MuleApplicationStructureGenerator(Path baseFolderPath, Path sourceFolder) {

    checkArgument(baseFolderPath != null, "The base folder must not be null");
    checkArgument(sourceFolder != null, "The source folder must not be null");

    this.baseFolderPath = baseFolderPath;
    this.originFolderPath = sourceFolder;
  }

  public MuleApplicationStructureGenerator withMavenComponents(MavenComponents mavenComponents) {
    checkArgument(mavenComponents != null, "The maven components must not be null");
    this.mavenComponents = mavenComponents;
    return this;
  }

  public MuleApplicationStructureGenerator isHeavyWeight(boolean heavyWeight) {
    this.heavyWeight = heavyWeight;
    return this;
  }

  public MuleApplicationStructureGenerator withPackageBuilder(MulePackageBuilder packageBuilder) {
    checkArgument(packageBuilder != null, "The package builder must not be null");
    this.packageBuilder = packageBuilder;
    return this;
  }

  public MuleApplicationStructureGenerator withDependencies(List<BundleDependency> dependencies) {
    checkArgument(dependencies != null, "The dependencies must not be null");
    this.dependencies = dependencies;
    return this;
  }

  public MuleApplicationStructureGenerator withPom(Pom pom) {
    checkArgument(pom != null, "The pom must not be null");
    this.pom = pom;
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Path generate(Path destinationFolder, RunConfiguration runConfiguration) throws Exception {
    checkArgument(destinationFolder != null, "The destination folder must not be null");
    PackagingOptions packagingOptions = buildPackagingOptions(runConfiguration);
    createExplodedPackage(destinationFolder, runConfiguration, packagingOptions);
    reGenerateMuleArtifactJson(destinationFolder, packagingOptions);

    return destinationFolder;
  }

  private PackagingOptions buildPackagingOptions(RunConfiguration runConfiguration) {
    MuleVersion runtimeVersion = new MuleVersion(runConfiguration.getContainerConfiguration().getRuntimeId());
    boolean isLightweight = !heavyWeight;
    boolean useLocalRepository = !heavyWeight && runtimeVersion.atLeast(LOCAL_REPOSITORY_MIN_MULE_VERSION);
    return new PackagingOptions(false, isLightweight, false, true, useLocalRepository);
  }

  private void createExplodedPackage(Path destinationFolderPath, RunConfiguration runConfiguration,
                                     PackagingOptions packagingOptions)
      throws Exception {
    MuleArchiver archiver = new MuleExplodedArchiver();
    archiver.getArchiver().setDuplicateBehavior(DUPLICATES_ADD);

    Path workingDir = destinationFolderPath.getParent();
    Path repositoryLocation = workingDir.resolve(REPOSITORY.value());

    FileUtils.copyDirectoryToDirectory(originFolderPath.resolve(FolderNames.META_INF.value()).toFile(), workingDir.toFile());

    if (mavenComponents != null) {
      SourcesProcessor processSources = new SourcesProcessor(mavenComponents);
      processSources.process(true, packagingOptions.isLightweightPackage(), packagingOptions.isUseLocalRepository(),
                             packagingOptions.isTestPackage(), workingDir.toFile(),
                             workingDir.resolve(FolderNames.META_INF.value()).resolve(FolderNames.MULE_ARTIFACT.value())
                                 .toFile(),
                             empty());
    }

    Path metaInfPath = workingDir.resolve(FolderNames.META_INF.value());

    packageBuilder
        .withPackagingOptions(packagingOptions)
        .withArchiver(archiver)
        .withClasses(originFolderPath.resolve(CLASSES.value()).toFile())
        .withMaven(metaInfPath.resolve(MAVEN.value()).toFile())
        .withMuleArtifact(metaInfPath.resolve(FolderNames.MULE_ARTIFACT.value()).toFile())
        .withTestClasses(originFolderPath.resolve(TEST_CLASSES.value()).toFile())
        .withTestMule(originFolderPath.resolve(TEST_MULE.value()).resolve(MUNIT.value()).toFile());

    if (!packagingOptions.isLightweightPackage()) {
      packageBuilder.withRepository(repositoryLocation.toFile());
    }

    if (packagingOptions.isAttachMuleSources()) {
      packageBuilder.withMuleSrc(metaInfPath.resolve(MULE_SRC.value()).toFile());
    }

    packageBuilder.createPackage(destinationFolderPath);

    File runConfigurationFile = destinationFolderPath.resolve(RUN_CONFIGURATION_PATH).toFile();
    writeStringToFile(runConfigurationFile, gson.toJson(runConfiguration), defaultCharset());
  }

  protected void reGenerateMuleArtifactJson(Path destinationFolderPath, PackagingOptions packagingOptions) throws IOException {
    Path muleArtifactFolderPath = destinationFolderPath.resolve(META_INF.value()).resolve(MULE_ARTIFACT.value());

    ProjectStructure projectStructure = new ProjectStructure(baseFolderPath, packagingOptions.isTestPackage());

    MuleArtifactContentResolver muleArtifactContentResolver =
        new MuleArtifactContentResolver(projectStructure, pom, dependencies);

    new DefaultValuesMuleArtifactJsonGenerator().generate(muleArtifactFolderPath, muleArtifactFolderPath,
                                                          muleArtifactContentResolver);
  }

}
