/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.maven.plugin.logging.Log;
import org.apache.maven.shared.utils.cli.*;

/**
 * <p>
 * The goal of this class is to start a new JVM based on the given parameters
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class JVMStarter {

  private static final Pattern JAVA_VERSION_PATTERN = Pattern.compile("java\\.version\\s+=\\s+(\\S+)\\s+");

  private Log log;
  protected Commandline commandline = new Commandline();
  protected Commandline commandlineVersion = new Commandline();


  public JVMStarter(Log log) throws IOException {
    this.log = log;
  }

  public JVMStarter withJVM(String jvm) {
    commandline.setExecutable(jvm);
    commandlineVersion.setExecutable(jvm);
    return this;
  }

  public JVMStarter withJar(File jarFile) {
    commandline.createArg().setValue("-jar");
    commandline.createArg().setFile(jarFile);
    return this;
  }

  public JVMStarter withWorkingDirectory(File workingDirectory) {
    commandline.setWorkingDirectory(workingDirectory);
    commandlineVersion.setWorkingDirectory(workingDirectory);
    return this;
  }

  public JVMStarter withArgLines(List<String> argLines) {
    for (String argLine : argLines) {
      commandline.createArg(true).setValue(argLine);
    }
    return this;
  }

  public JVMStarter withArgLines(Map<String, File> argLines) {
    for (Map.Entry<String, File> argLine : argLines.entrySet()) {
      commandline.createArg().setValue(argLine.getKey());
      commandline.createArg().setFile(argLine.getValue());
    }
    return this;
  }

  public JVMStarter withSystemProperties(Map<String, String> systemProperties) {
    for (Map.Entry<String, String> prop : systemProperties.entrySet()) {
      commandline.createArg(true).setValue("-D" + prop.getKey() + "=" + prop.getValue());
    }
    return this;
  }

  public JVMStarter addEnvironmentVariables(Map<String, String> environmentVariables) {
    for (Map.Entry<String, String> envVar : environmentVariables.entrySet()) {
      commandline.addEnvironment(envVar.getKey(), envVar.getValue());
    }
    return this;
  }

  public int execute(StreamConsumer outStreamConsumer, StreamConsumer errStreamConsumer) throws CommandLineException {
    return CommandLineUtils.executeCommandLine(commandline, outStreamConsumer, errStreamConsumer);
  }

  public String getVersion() {
    outputStreamString outStream = new outputStreamString();
    outputStreamString outStreamError = new outputStreamString();
    try {
      commandlineVersion.addArguments("-XshowSettings:properties", "-version");
      CommandLineUtils.executeCommandLine(commandlineVersion, outStream, outStreamError);
    } catch (CommandLineException e) {
      throw new RuntimeException(e);
    }
    String output = outStream.getOutput().equals("") ? outStreamError.getOutput() : outStream.getOutput();

    Matcher matcher = JAVA_VERSION_PATTERN.matcher(output);
    if (matcher.find()) {
      return matcher.group(1);
    }

    return "";
  }

}


class outputStreamString implements StreamConsumer {

  private String output = "";

  @Override
  public void consumeLine(String line) throws IOException {
    output = output + line;
  }

  public String getOutput() {
    return output;
  }
}

