/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven;

import static de.skuzzle.semantic.Version.parseVersion;
import static java.io.File.pathSeparator;
import static java.lang.String.join;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.mule.munit.plugin.maven.runtime.DiscoverProduct.DISCOVER_PRODUCT;
import static org.mule.munit.plugin.maven.runtime.RuntimeProducts.EE;
import static org.mule.munit.plugin.maven.project.MuleApplicationStructureGenerator.RUN_CONFIGURATION_JSON;

import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.io.FileUtils.readFileToString;
import static org.mule.munit.remote.classloading.ClassLoaderUtils.STARTER_JAR_FILE_NAME;

import com.google.common.collect.ImmutableList;
import org.apache.maven.model.Dependency;
import org.apache.maven.plugin.MojoExecution;
import org.apache.maven.plugin.PluginParameterExpressionEvaluator;
import org.mule.munit.common.util.FileUtils;
import org.mule.munit.plugin.maven.fips.FipsArgumentsResolver;
import org.mule.munit.plugin.maven.locators.JVMLocator;
import org.mule.munit.plugin.maven.locators.ProductVersionsLocator;
import org.mule.munit.plugin.maven.locators.RemoteRepositoriesLocator;
import org.mule.munit.plugin.maven.locators.TestSuiteFilesLocator;
import org.mule.munit.plugin.maven.runner.JVMStarter;
import org.mule.munit.plugin.maven.runner.MessageHandlerFactory;
import org.mule.munit.plugin.maven.runner.consumer.ErrorStreamConsumer;
import org.mule.munit.plugin.maven.runner.consumer.RunnerStreamConsumer;
import org.mule.munit.plugin.maven.runner.model.Debugger;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.printer.ResultPrinter;
import org.mule.munit.plugin.maven.runner.structure.WorkingDirectoryGenerator;
import org.mule.munit.plugin.maven.runtime.DiscoverProduct;
import org.mule.munit.plugin.maven.runtime.Product;
import org.mule.munit.plugin.maven.runtime.ProductConfiguration;
import org.mule.munit.plugin.maven.runtime.RuntimeConfiguration;
import org.mule.munit.plugin.maven.runtime.TargetProduct;
import org.mule.munit.plugin.maven.util.ArgLinesManager;
import org.mule.munit.plugin.maven.util.ClasspathManager;
import org.mule.munit.plugin.maven.util.JarFileFactory;
import org.mule.munit.plugin.maven.util.MuleApplicationModelLoader;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.plugin.maven.util.RuntimeVersionProviderFactory;
import org.mule.munit.plugin.maven.util.properties.UserPropertiesBuilder;
import org.mule.munit.remote.RemoteRunner;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.RunConfigurationParser;
import org.mule.munit.plugin.maven.project.ApplicationStructureGenerator;

import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.ProjectBuilder;
import org.apache.maven.shared.utils.cli.CommandLineException;
import org.apache.maven.toolchain.ToolchainManager;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipEntry;

/**
 * Base Mojo to run MUnit tests
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public abstract class AbstractMunitMojo extends AbstractMojo {

  private static final List<String> MODULES = ImmutableList.of(
                                                               "java.base/java.lang.invoke=munit.starter",
                                                               "java.base/java.lang.reflect=munit.starter",
                                                               "java.base/java.lang=munit.starter",
                                                               "java.sql/java.sql=munit.starter",
                                                               "java.base/sun.security.provider=org.bouncycastle.fips.core");



  protected static final String ARG_TOKEN = "-";
  protected static final String RUN_CONFIGURATION_ARG = ARG_TOKEN + RunConfigurationParser.RUN_CONFIGURATION_PARAMETER;

  public static final String STARTER_CLASS_FILE = "munitstarter";

  public static final String MUNIT_PREVIOUS_RUN_PLACEHOLDER = "MUNIT_PREVIOUS_RUN_PLACEHOLDER";

  protected static final Class REMOTE_RUNNER_CLASS = RemoteRunner.class;
  private static final String SKIP_TESTS_PROPERTY = "skipTests";
  private static final String SKIP_MUNIT_TESTS_PROPERTY = "skipMunitTests";

  public static final String MULE_ARTIFACT_JSON_FILE_NAME = "mule-artifact.json";

  @Parameter(property = "munit.test")
  public String munitTest;

  @Parameter(property = "munit.tags")
  public String munitTags;

  @Parameter(property = "munit.failIfNoTests", defaultValue = "true")
  public boolean munitFailIfNoTests = true;

  @Parameter(property = "munit.debug", defaultValue = "false")
  public String munitDebug;

  @Parameter(property = "runtimeVersion")
  public String runtimeVersion;

  @Parameter(property = "runtimeProduct")
  public String runtimeProduct;

  @Parameter(property = "runtimeLocalDistribution")
  public String runtimeLocalDistribution;

  @Parameter(defaultValue = "${skipMunitTests}")
  public boolean skipMunitTests = false;

  @Parameter(defaultValue = "false")
  public boolean clearParameters = false;

  @Parameter(defaultValue = "${project.build.directory}/test-mule/munit")
  public File munitTestsDirectory;

  @Parameter(defaultValue = "${project.build.directory}/test-mule/munit", readonly = true)
  public File defaultMunitTestsDirectory;

  @Parameter(property = "project", required = true)
  public MavenProject project;

  @Parameter(defaultValue = "${session}", readonly = true)
  public MavenSession session;

  @Parameter(defaultValue = "${repositorySystemSession}", readonly = true)
  public RepositorySystemSession repositorySystemSession;

  @Parameter(defaultValue = "${plugin.artifactId}")
  public String pluginArtifactId;

  @Parameter(property = "redirectTestOutputToFile", defaultValue = "false")
  public boolean redirectTestOutputToFile;

  @Parameter(defaultValue = "${project.build.directory}/munit-reports/output/")
  public File testOutputDirectory;

  @Parameter(property = "enableSurefireReports", defaultValue = "true")
  public boolean enableSurefireReports;

  @Parameter(property = "enableSonarReports", defaultValue = "true")
  public boolean enableSonarReports;

  @Parameter(defaultValue = "${project.build.directory}/surefire-reports/")
  public File surefireReportsFolder;

  @Parameter(defaultValue = "${project.build.directory}/sonar-reports/")
  public File sonarReportsFolder;

  @Parameter(property = "skipAfterFailure", defaultValue = "false")
  public boolean skipAfterFailure = false;

  @Parameter(property = "munit.jvm")
  public String jvm;

  @Parameter(property = "munit.addFipsProviders", defaultValue = "false")
  public boolean addFipsProviders;

  @Parameter(property = "munit.fipsType", defaultValue = "false")
  public String fipsType;

  @Parameter(property = "munit.addBootDependencies")
  public String addBootDependencies;

  @Parameter(property = "munit.addOpens")
  public List<String> addOpens;

  @Parameter(property = "additionalFipsProviders")
  public List<Dependency> additionalFipsDependencies;

  @Parameter(property = "securityPropertiesFile")
  public String securityPropertiesFile;

  @Parameter(property = "argLines")
  public List<String> argLines;

  @Parameter(property = "system.property.variables")
  public Map<String, String> systemPropertyVariables;

  @Parameter(property = "environment.variables")
  public Map<String, String> environmentVariables;

  @Parameter(property = "dynamic.ports")
  public List<String> dynamicPorts;

  /**
   * Set this to true to ignore a failure during testing. Its use is NOT RECOMMENDED, but quite convenient on occasion.
   */
  @Parameter(property = "maven.test.failure.ignore", defaultValue = "false")
  public boolean testFailureIgnore;

  /**
   * @deprecated since 2.4 use {@link #productConfiguration} instead.
   */
  @Parameter
  @Deprecated
  public RuntimeConfiguration runtimeConfiguration;

  @Parameter
  public ProductConfiguration productConfiguration;

  @Component
  protected ToolchainManager toolchainManager;

  @Component
  public RepositorySystem repositorySystem;

  @Component
  protected ProjectBuilder projectBuilder;

  @Component
  protected org.apache.maven.repository.RepositorySystem mavenRepositorySystem;

  @Parameter(property = "project.build.directory", required = true)
  public File outputDirectory;

  @Parameter(readonly = true, required = true, defaultValue = "${localRepository}")
  public ArtifactRepository localRepository;

  @Parameter(readonly = true, required = true, defaultValue = "${project.remoteArtifactRepositories}")
  public List<ArtifactRepository> remoteArtifactRepositories;

  @Parameter(defaultValue = "${project.basedir}")
  public File projectBaseFolder;


  @Component
  protected MojoExecution execution;
  protected PluginParameterExpressionEvaluator expressionEvaluator;

  protected Gson gson = new GsonBuilder().setPrettyPrinting().create();

  /**
   * Processes argLines to support JSON arrays.
   * If argLines contains a string starting with '[', it will be parsed as a JSON array.
   * This allows passing multiple arguments from command line like:
   * -DargLines='["-Xmx2g", "-XX:+UseG1GC", "-Dmule.verbose.exceptions=true"]'
   */
  protected List<String> processArgLines(List<String> argLines) {
    if (argLines == null || argLines.isEmpty()) {
      return new ArrayList<>();
    }

    List<String> processedArgLines = new ArrayList<>();

    for (String argLine : argLines) {
      if (argLine != null && argLine.trim().startsWith("[")) {
        try {
          // Parse as JSON array
          String[] jsonArray = gson.fromJson(argLine, String[].class);
          for (String item : jsonArray) {
            if (item != null && !item.trim().isEmpty()) {
              processedArgLines.add(item.trim());
            }
          }
          getLog().debug("Parsed JSON array for argLines: " + argLine);
        } catch (Exception e) {
          getLog().warn("Failed to parse argLines as JSON array: " + argLine + ". Using as single argument. Error: "
              + e.getMessage());
          processedArgLines.add(argLine);
        }
      } else if (argLine != null && !argLine.trim().isEmpty()) {
        processedArgLines.add(argLine.trim());
      }
    }

    return processedArgLines;
  }

  public List<File> suiteFiles;
  public JVMLocator jvmLocator;
  public JarFileFactory jarFileFactory;
  public ClasspathManager classpathManager;
  public MessageHandlerFactory messageHandlerFactory;
  public Map<String, String> effectiveSystemProperties;
  public WorkingDirectoryGenerator workingDirectoryGenerator;
  public MuleApplicationModelLoader muleApplicationModelLoader;
  public RuntimeVersionProviderFactory runtimeVersionProviderFactory;
  public ProductVersionsLocator productVersionsLocator;
  public ResultPrinterFactory resultPrinterFactory;
  private String starterJarFileName = "";
  protected Debugger debugger;
  private FipsArgumentsResolver fipsArgumentsResolver;

  protected void init() throws MojoExecutionException {
    this.expressionEvaluator = new PluginParameterExpressionEvaluator(session, execution);
    this.jarFileFactory = new JarFileFactory();
    this.starterJarFileName = FileUtils.generateRandomFileName(STARTER_CLASS_FILE, ".jar");
    this.messageHandlerFactory = new MessageHandlerFactory(getLog());
    this.validateProperties();
    this.effectiveSystemProperties = getEffectiveSystemProperties();
    this.jvmLocator = new JVMLocator(session, jvm, toolchainManager, getLog());
    this.workingDirectoryGenerator = new WorkingDirectoryGenerator(getLog(), getApplicationStructureGenerator(), project);
    this.classpathManager = new ClasspathManager(REMOTE_RUNNER_CLASS, getLog());
    this.muleApplicationModelLoader = getMuleApplicationModelLoader();
    this.resultPrinterFactory = getResultPrinterFactory();

    initConfiguration();

    this.runtimeVersionProviderFactory =
        new RuntimeVersionProviderFactory(repositorySystem, repositorySystemSession, createRemoteRepositoriesLocator(), getLog());
    this.productVersionsLocator = new ProductVersionsLocator(runtimeVersionProviderFactory, getLog());
    this.debugger = Debugger.fromString(munitDebug);

    if (addFipsProviders) {
      try {
        fipsArgumentsResolver = new FipsArgumentsResolver(getLog(), repositorySystem, repositorySystemSession,
                                                          project.getRemoteProjectRepositories(),
                                                          additionalFipsDependencies, securityPropertiesFile, fipsType);
      } catch (IOException e) {
        throw new MojoExecutionException(e);
      }
    }
  }

  protected void initConfiguration() {
    if (this.runtimeConfiguration != null) {
      // Keep backwards compatibility
      this.productConfiguration = ProductConfiguration.builder()
          .from(productConfiguration)
          .withRuntimeConfiguration(RuntimeConfiguration.builder()
              .from(runtimeConfiguration)
              .build())
          .withOverrides(session.getUserProperties())
          .build();
    } else {
      this.productConfiguration = ProductConfiguration.builder()
          .from(productConfiguration)
          .withOverrides(session.getUserProperties())
          .build();
    }
    this.runtimeConfiguration = null;
  }

  @Override
  public void execute() throws MojoExecutionException {
    if (!"true".equals(System.getProperty(SKIP_TESTS_PROPERTY))) {
      if (!skipMunitTests) {
        if (!hasExecutedBefore()) {
          if (hasSuites()) {
            init();
            doExecute();
          } else {
            getLog().warn("MUnit will not run, no MUnit suites found in your project");
          }
        } else {
          getLog().info("Skipping execution of munit because it has already been run");
        }
      } else {
        getLog().info(format("Run of %s skipped. Property [%s] was set to true", pluginArtifactId, SKIP_MUNIT_TESTS_PROPERTY));
      }
    } else {
      getLog().info(format("Run of %s skipped. Property [%s] was set to true", pluginArtifactId, SKIP_TESTS_PROPERTY));
    }
  }

  /**
   * This method avoids running MUnit more than once. This could happen because other phases such as the coverage-report, can
   * trigger the "test" phase since it needs its information.
   *
   * @return true if the MUnit run has already happened before
   */
  @SuppressWarnings("unchecked")
  protected boolean hasExecutedBefore() {
    Map<String, String> pluginContext = getPluginContext();
    if (pluginContext.containsKey(MUNIT_PREVIOUS_RUN_PLACEHOLDER)) {
      return true;
    }
    getPluginContext().put(MUNIT_PREVIOUS_RUN_PLACEHOLDER, MUNIT_PREVIOUS_RUN_PLACEHOLDER);
    return false;
  }

  public void doExecute() throws MojoExecutionException {

    Map<TargetProduct, RunResult> runResults = new TreeMap<>();
    try {
      Map<TargetProduct, RunConfiguration> runConfigurations = getRunConfigurations();
      for (Map.Entry<TargetProduct, RunConfiguration> runConfigurationEntry : runConfigurations.entrySet()) {

        RunConfiguration runConfiguration = runConfigurationEntry.getValue();
        workingDirectoryGenerator.generate(runConfiguration);

        TargetProduct targetProduct = runConfigurationEntry.getKey();
        getLog().info(format("Running %s with version %s", targetProduct.getProduct(), targetProduct.getVersion()));

        if (runConfiguration.getSuitePaths().isEmpty()) {
          if (munitFailIfNoTests) {
            throw new MojoExecutionException("No tests suites were found! (Set -Dmunit.failIfNoTests=false to ignore this error.)");
          } else {
            getLog().info("No MUnit test suite files found. No test will be run");
            return;
          }
        }

        RunnerStreamConsumer streamConsumer = new RunnerStreamConsumer(messageHandlerFactory.create(redirectTestOutputToFile));
        streamConsumer.setRunConfiguration(runConfiguration);
        ErrorStreamConsumer errorStreamConsumer = new ErrorStreamConsumer(redirectTestOutputToFile);
        int result = createTestRunExecutor(runConfiguration).execute(streamConsumer, errorStreamConsumer);

        RunResult runResult;
        if (result == 0) {
          runResult = streamConsumer.getRunResult();
          runResults.put(targetProduct, runResult);
        } else {
          throw new MojoExecutionException("Build Failed", new MojoExecutionException(errorStreamConsumer.getOutput()));
        }

      }
      runResults.forEach(this::handleRunResult);
      failBuildIfNecessary(runResults);

    } catch (IOException | CommandLineException e) {
      throw new MojoExecutionException("Build Failed", e);
    }
  }

  protected JVMStarter createTestRunExecutor(RunConfiguration runConfiguration) throws IOException, MojoExecutionException {
    File runConfigurationFile = saveRunConfigurationToFile(runConfiguration);
    File log4jConfigurationFile = new File(runConfiguration.getContainerConfiguration().getLog4JConfigurationFilePath());
    return createJVMStarter(runConfigurationFile, getEffectiveArgLines(starterJarFileName, log4jConfigurationFile),
                            runConfiguration.getContainerConfiguration().getRuntimeId());
  }

  protected void handleRunResult(TargetProduct targetProduct, RunResult runResult) {
    for (ResultPrinter printer : resultPrinterFactory.create()) {
      printer.print(targetProduct, runResult);
    }
  }

  protected void failBuildIfNecessary(Map<TargetProduct, RunResult> runResults) throws MojoExecutionException {
    if (runResults.values().stream().anyMatch(RunResult::finishedWithErrors)) {
      throw new MojoExecutionException("There was an error running MUnit tests");
    }

    if (!testFailureIgnore && runResults.values().stream().anyMatch(RunResult::hasFailed)) {
      throw new MojoExecutionException("MUnit Tests Failed");
    }

    if (munitFailIfNoTests && runResults.values().stream().allMatch(result -> result.getNumberOfTests() == 0)) {
      throw new MojoExecutionException("No tests were executed! (Set -Dmunit.failIfNoTests=false to ignore this error.)");
    }
  }

  protected abstract ResultPrinterFactory getResultPrinterFactory();

  protected abstract ApplicationStructureGenerator getApplicationStructureGenerator() throws MojoExecutionException;

  protected abstract File getMuleApplicationJsonPath();

  protected abstract RunConfiguration createRunConfiguration(TargetProduct targetProduct) throws MojoExecutionException;

  /**
   * @since 2.4
   */
  protected RunConfiguration createRunConfiguration(TargetProduct targetRuntime, String additionalTags)
      throws MojoExecutionException {
    // Default implementation to keep backwards compatibility with extending classes
    return createRunConfiguration(targetRuntime);
  }

  protected Map<TargetProduct, RunConfiguration> getRunConfigurations() throws MojoExecutionException, IOException {
    Map<TargetProduct, RunConfiguration> runConfigurations = new TreeMap<>();
    List<TargetProduct> effectiveTargetRuntimes = new ArrayList<>();

    if (shouldRunSingleRuntime()) {
      addDefaultTargetProducts(effectiveTargetRuntimes);
    } else {
      List<String> additionalProducts = productConfiguration.getAdditionalProducts();

      List<DiscoverProduct> productsToDiscover = productsToDiscover();

      if (additionalProducts != null && !additionalProducts.isEmpty()) {
        effectiveTargetRuntimes.addAll(additionalProducts.stream().map(TargetProduct::parse).collect(toList()));
      }
      if (!productsToDiscover.isEmpty()) {
        for (DiscoverProduct discoverProduct : productsToDiscover) {
          effectiveTargetRuntimes.addAll(discoverProducts(muleApplicationModelLoader.getRuntimeVersion(), discoverProduct));
        }
      } else {
        addDefaultTargetProducts(effectiveTargetRuntimes);
      }
    }

    for (TargetProduct targetRuntime : effectiveTargetRuntimes) {
      RunConfiguration runConfiguration = createRunConfiguration(targetRuntime);
      runConfigurations.put(targetRuntime, runConfiguration);
    }
    return runConfigurations;
  }

  protected List<DiscoverProduct> productsToDiscover() {
    return productConfiguration.getDiscoverProducts()
        .stream()
        .filter(discoverProduct -> !discoverProduct.isSkipped())
        .collect(toList());
  }

  protected boolean shouldRunSingleRuntime() {
    return runtimeVersion != null || runtimeProduct != null || runtimeLocalDistribution != null;
  }

  private Set<TargetProduct> discoverProducts(String version, DiscoverProduct discoverProduct)
      throws MojoExecutionException {
    Product product = discoverProduct.getProductId()
        .orElseThrow(() -> new MojoExecutionException(format("Invalid Product Configuration: the auto discovery of products was "
            + "enabled but no product kind was specified, neither in the plugin nor using the command line argument '%s'. One of "
            + "the following products has to be selected: %s",
                                                             DISCOVER_PRODUCT,
                                                             asList(Product.values()))));

    if (muleApplicationModelLoader.getRuntimeProduct().equals(EE.value()) && !product.supportsEe()) {
      throw new MojoExecutionException(format("Product is EE only but was configured to discover %s runtimes.",
                                              product.name()));
    }

    productVersionsLocator
        .includingSnapshots(discoverProduct.isIncludeSnapshots())
        .withMinVersion(discoverProduct.getMinVersion().orElse(version))
        .withProduct(product)
        .usingLatestPatches(discoverProduct.isUseLatestPatches());

    return productVersionsLocator.locate();
  }

  protected void addDefaultTargetProducts(List<TargetProduct> effectiveTargetProducts)
      throws IOException, MojoExecutionException {
    effectiveTargetProducts
        .add(new TargetProduct(muleApplicationModelLoader.getRuntimeVersion(),
                               muleApplicationModelLoader.getRuntimeProduct().equals("MULE")
                                   ? Product.MULE_CE
                                   : Product.MULE_EE));
  }

  public List<String> getEffectiveArgLines(String starterJarFileName, File log4jConfigurationFile) {
    // Process argLines to support JSON arrays
    List<String> processedArgLines = processArgLines(argLines);

    if (!Optional.ofNullable(debugger).map(Debugger::getAddDebugger).orElse(false)) {
      return new ArgLinesManager(processedArgLines, starterJarFileName, munitDebug, log4jConfigurationFile, getLog())
          .getEffectiveArgLines();
    }

    List<String> effectiveArgLines =
        new ArgLinesManager(processedArgLines, starterJarFileName, null, log4jConfigurationFile, getLog()).getEffectiveArgLines();
    return Stream
        .concat(effectiveArgLines.stream(), debugger.getArguments().stream())
        .filter(StringUtils::isNotBlank)
        .collect(toList());
  }

  public JVMStarter createJVMStarter(File runConfigurationFile, List<String> argLines, String runtimeVersion)
      throws IOException, MojoExecutionException {
    getLog().debug("MUnit root folder found at: " + munitTestsDirectory.getAbsolutePath());

    JVMStarter jvmStarter = new JVMStarter(getLog())
        .withJVM(jvmLocator.locate())
        .withWorkingDirectory(getWorkingDirectory());

    String javaVersion = jvmStarter.getVersion();
    List<String> args = new ArrayList<String>();
    Consumer<String> addOpenModule = module -> {
      args.add(module);
      args.add("--add-opens");
      getLog().info("--add-opens: " + module);
    };
    Map<String, File> fileArgLines = new HashMap<>();
    if (javaVersion != null && ((javaVersion.startsWith("17") || javaVersion.startsWith("21")
        || (javaVersion.startsWith("11") && modularizeRuntime(runtimeVersion))))) {
      unzipFile("munit-remote");
      unzipFile("munit-starter");
      List<String> modulePath = new ArrayList<>();

      if (addFipsProviders) {
        modulePath.addAll(fipsArgumentsResolver.getClassPathEntriesForFipsProviders());
      }
      if (addBootDependencies != null) {
        modulePath.add(addBootDependencies);
      }

      args.add(outputDirectory.toPath().resolve("munit-remote-" + getVersion()).resolve("lib").toString());
      args.add("-module_path");
      args.add("munit.starter/org.mule.munit.starter.LayerStarter");
      args.add("-m");

      modulePath.add(outputDirectory.toPath().resolve("munit-starter-" + getVersion()).resolve("lib").toString());
      args.add(join(pathSeparator, modulePath));
      args.add("--module-path");

      MODULES.forEach(addOpenModule);
      if (addOpens != null) {
        addOpens.forEach(addOpenModule);
      }
    } else {
      List<String> classPath = classpathManager.getEffectiveClasspath();
      if (addFipsProviders) {
        classPath.addAll(fipsArgumentsResolver.getClassPathEntriesForFipsProviders());
      }
      File jarFile = jarFileFactory.create(classPath, REMOTE_RUNNER_CLASS.getCanonicalName(),
                                           new File(project.getBuild().getDirectory()), starterJarFileName);
      jvmStarter.withJar(jarFile);
      argLines.add("-D" + STARTER_JAR_FILE_NAME + "=" + starterJarFileName);
    }

    if (addFipsProviders) {
      argLines.add(fipsArgumentsResolver.getSecurePropertiesArgument());
    }

    fileArgLines.put(RUN_CONFIGURATION_ARG, runConfigurationFile);
    jvmStarter.withArgLines(args).withArgLines(fileArgLines).withArgLines(argLines)
        .withSystemProperties(effectiveSystemProperties).addEnvironmentVariables(environmentVariables);

    return jvmStarter;
  }

  protected abstract String getVersion();

  private boolean modularizeRuntime(String runtimeVersion) {
    Matcher matcher = Pattern.compile("^(\\d+\\.\\d+\\.\\d+).*$").matcher(runtimeVersion);

    if (matcher.matches()) {
      return parseVersion(matcher.group(1)).compareTo(parseVersion("4.6.0")) >= 0;
    }

    throw new IllegalArgumentException("Invalid runtimeVersion: " + runtimeVersion);
  }


  protected void unzipFile(String module) throws MojoExecutionException {
    try {
      String fileZip =
          Paths.get((this.localRepository).getBasedir()).resolve("com")
              .resolve("mulesoft")
              .resolve("munit").resolve(module).resolve(getVersion())
              .resolve(module + "-" + getVersion() + "-dependencies.zip")
              .toString();
      byte[] buffer = new byte[1024];
      ZipInputStream zis = new ZipInputStream(new FileInputStream(fileZip));
      ZipEntry zipEntry = zis.getNextEntry();
      while (zipEntry != null) {
        File newFile = newFile(outputDirectory, zipEntry);
        if (zipEntry.isDirectory()) {
          if (!newFile.isDirectory() && !newFile.mkdirs()) {
            throw new IOException("Failed to create directory " + newFile);
          }
        } else {
          // fix for Windows-created archives
          File parent = newFile.getParentFile();
          if (!parent.isDirectory() && !parent.mkdirs()) {
            throw new IOException("Failed to create directory " + parent);
          }

          // write file content
          FileOutputStream fos = new FileOutputStream(newFile);
          int len;
          while ((len = zis.read(buffer)) > 0) {
            fos.write(buffer, 0, len);
          }
          fos.close();
        }
        zipEntry = zis.getNextEntry();
      }

      zis.closeEntry();
      zis.close();
    } catch (Exception e) {
      throw new MojoExecutionException(e.getMessage(), e);
    }
  }

  public static File newFile(File destinationDir, ZipEntry zipEntry) throws IOException {
    File destFile = new File(destinationDir, zipEntry.getName());

    String destDirPath = destinationDir.getCanonicalPath();
    String destFilePath = destFile.getCanonicalPath();

    if (!destFilePath.startsWith(destDirPath + File.separator)) {
      throw new IOException("Entry is outside of the target dir: " + zipEntry.getName());
    }

    return destFile;
  }

  public File saveRunConfigurationToFile(RunConfiguration runConfiguration) throws IOException {
    File runConfigurationFile =
        Paths.get(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath(), RUN_CONFIGURATION_JSON)
            .toFile();
    saveAsJsonDataToFile(runConfiguration, runConfigurationFile);
    return runConfigurationFile;
  }

  protected void saveAsJsonDataToFile(Object data, File dataLocation) {
    try {
      if (dataLocation == null) {
        getLog().warn("Unable to save data, no destination file was provided");
        return;
      }
      dataLocation.getParentFile().mkdirs();
      dataLocation.createNewFile();
      writeStringToFile(dataLocation, gson.toJson(data), defaultCharset());
      getLog().debug("Data File saved in " + dataLocation);
    } catch (IOException e) {
      getLog().warn("Unable to save data to file:" + e.getMessage());
      getLog().debug(e);
    }
  }

  protected MuleApplicationModelLoader getMuleApplicationModelLoader() throws MojoExecutionException {
    return new MuleApplicationModelLoader(getMuleApplicationModel(), getLog()).withRuntimeProduct(runtimeProduct)
        .withRuntimeVersion(runtimeVersion).withRuntimeLocalDistribution(runtimeLocalDistribution);
  }

  protected MuleApplicationModel getMuleApplicationModel() throws MojoExecutionException {
    File muleApplicationJsonPath = getMuleApplicationJsonPath();
    try {
      return new MuleApplicationModelJsonSerializer().deserialize(readFileToString(muleApplicationJsonPath, defaultCharset()));
    } catch (IOException e) {
      String message = "Fail to read mule application file from " + muleApplicationJsonPath;
      getLog().error(message, e);
      throw new MojoExecutionException(message, e);
    }
  }

  public File getWorkingDirectory() {
    if (project != null) {
      return project.getBasedir();
    }
    return new File(".");
  }

  private Map<String, String> getEffectiveSystemProperties() {
    Map<String, String> effectiveSystemProperties = new HashMap<>();
    effectiveSystemProperties.putAll(getUserSystemProperties());
    // TODO MU-978
    effectiveSystemProperties.put("org.glassfish.grizzly.DEFAULT_MEMORY_MANAGER",
                                  "org.glassfish.grizzly.memory.HeapMemoryManager");
    return effectiveSystemProperties;
  }

  private Map<String, String> getUserSystemProperties() {
    UserPropertiesBuilder builder = new UserPropertiesBuilder(project.getBuild().getDirectory(), getLog())
        .withSystemPropertyVariables(systemPropertyVariables)
        .withDynamicPorts(dynamicPorts)
        .withUserProperties(session != null ? session.getUserProperties() : null);

    return builder.build();
  }

  private boolean hasSuites() {
    suiteFiles = new TestSuiteFilesLocator().locateFiles(munitTestsDirectory);
    return !suiteFiles.isEmpty();
  }

  protected RemoteRepositoriesLocator createRemoteRepositoriesLocator() {
    return new RemoteRepositoriesLocator(project);
  }


  public void validateProperties() throws MojoExecutionException {

    if (!FileUtils.isInDirectory(defaultMunitTestsDirectory, munitTestsDirectory)) {
      throw new MojoExecutionException("munitTestsDirectory property must be a subfolder of ${project.build.directory}/test-mule/munit.");
    }
  }
}
