/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runtime;

import static org.mule.munit.plugin.maven.runtime.Product.MULE_CE;
import static org.mule.munit.plugin.maven.runtime.Product.MULE_EE;
import static org.mule.munit.plugin.maven.runtime.ProductDiscoveryMode.ALL;
import static org.mule.munit.plugin.maven.runtime.ProductDiscoveryMode.CE;
import static org.mule.munit.plugin.maven.runtime.ProductDiscoveryMode.EE;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.StringUtils.split;

import org.mule.munit.plugin.maven.AbstractMunitMojo;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

/**
 * Model of the {@link AbstractMunitMojo} product configuration parameter. It holds information about the products which to run
 * the tests against
 *
 * @author Mulesoft Inc.
 * @since 2.4.0
 */
public class ProductConfiguration {

  public static final String ADDITIONAL_PRODUCTS_PROPERTY = "additionalProducts";
  public static final String ADDITIONAL_PRODUCTS_SEPARATOR = ",";

  private List<DiscoverProduct> discoverProducts;
  private List<String> additionalProducts = emptyList();

  public ProductConfiguration() {}

  private ProductConfiguration(List<DiscoverProduct> discoverProducts,
                               List<String> additionalProducts) {
    this.discoverProducts = discoverProducts;
    this.additionalProducts = additionalProducts;
  }

  public List<DiscoverProduct> getDiscoverProducts() {
    return discoverProducts;
  }

  public List<String> getAdditionalProducts() {
    return additionalProducts;
  }

  public static Builder builder() {
    return new Builder();
  }

  /**
   * Builder for a {@link ProductConfiguration}
   *
   * @author Mulesoft Inc.
   * @since 2.4.0
   */
  public static class Builder {

    private List<DiscoverProduct> discoverProducts = new ArrayList<>();
    private List<String> additionalProducts = emptyList();

    public Builder from(ProductConfiguration productConfiguration) {
      if (productConfiguration == null) {
        return this;
      }

      discoverProducts = productConfiguration.getDiscoverProducts()
          .stream()
          .map(discoverProduct -> DiscoverProduct.builder().from(discoverProduct).build())
          .collect(toList());

      additionalProducts = productConfiguration.getAdditionalProducts();

      return this;
    }

    public Builder withRuntimeConfiguration(RuntimeConfiguration runtimeConfiguration) {
      if (runtimeConfiguration == null) {
        return this;
      }

      runtimeConfiguration.getDiscoverRuntimes()
          .ifPresent(discoverRuntimes -> {
            if (!discoverRuntimes.getProduct().isPresent()) {
              withRuntimeConfiguration(discoverRuntimes, null);
              return;
            }

            discoverRuntimes.getProduct().ifPresent(product -> {
              if (product == CE) {
                withRuntimeConfiguration(discoverRuntimes, MULE_CE);
              } else if (product == EE) {
                withRuntimeConfiguration(discoverRuntimes, MULE_EE);
              } else if (product == ALL) {
                withRuntimeConfiguration(discoverRuntimes, MULE_CE);
                withRuntimeConfiguration(discoverRuntimes, MULE_EE);
              }
            });
          });

      this.additionalProducts = runtimeConfiguration.getAdditionalRuntimes();

      return this;
    }

    private void withRuntimeConfiguration(DiscoverRuntimes discoverRuntimes, Product product) {
      org.mule.munit.plugin.maven.runtime.DiscoverProduct.Builder builder = DiscoverProduct.builder();
      discoverRuntimes.getMinMuleVersion().ifPresent(builder::withMinVersion);
      discoverProducts.add(builder.withProductId(product)
          .includingSnapshots(discoverRuntimes.isIncludeSnapshots())
          .skip(discoverRuntimes.isSkipped())
          .useLatestPatches(discoverRuntimes.isUseLatestPatches())
          .withRuntimeRepositories(discoverRuntimes.getRuntimeRepositories())
          .build());
    }

    public Builder withAdditionalProducts(List<String> additionalProducts) {
      this.additionalProducts = additionalProducts;
      return this;
    }

    public Builder withOverrides(Properties userProperties) {
      if (userProperties == null) {
        return this;
      }

      discoverProducts = discoverProducts
          .stream()
          .map(discoverProduct -> DiscoverProduct.builder().from(discoverProduct)
              .withOverrides(userProperties)
              .build())
          .collect(toList());

      if (userProperties.containsKey(ADDITIONAL_PRODUCTS_PROPERTY)) {
        this.additionalProducts = asList(split(userProperties.getProperty(ADDITIONAL_PRODUCTS_PROPERTY),
                                               ADDITIONAL_PRODUCTS_SEPARATOR));
      }

      return this;
    }

    public ProductConfiguration build() {
      return new ProductConfiguration(discoverProducts, additionalProducts);
    }
  }
}
