/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.config;

import  org.mule.munit.api.spring.config.parsers.MunitGenericDefinitionParser;
import org.mule.munit.api.spring.config.parsers.model.*;
import org.mule.munit.mock.MockModule;
import org.mule.munit.mock.model.Attribute;
import org.mule.munit.mock.model.MunitMuleMessage;
import org.mule.munit.mock.processors.*;
import org.mule.munit.mock.spring.config.AbstractMockNamespaceHandler;

import static org.mule.munit.api.spring.config.parsers.MunitGenericDefinitionParser.MunitGenericDefinitionParserBuilder;

/**
 * Registers bean definitions parsers for handling elements in <code>http://www.mulesoft.org/schema/mule/mock</code>.
 */
public class MockNamespaceHandler extends AbstractMockNamespaceHandler {

    public static final String CONFIG_MP = "config";
    public static final String OUTBOUND_ENDPOINT_MP = "outbound-endpoint";
    public static final String SPY_MP = "spy";
    public static final String THROW_AN_MP = "throw-an";
    public static final String VERIFY_CALL_MP = "verify-call";
    public static final String WHEN_MP = "when";

    public static final String CONFIG_REF_ATT_NAME = "config-ref";
    public static final String WITH_ATTRIBUTES_ATT_NAME = "with-attributes";

    protected void declareBeanDefinitionParsers() {
        declareBeanDefinitionParser(CONFIG_MP, buildMockModuleDefinitionParser());
        declareBeanDefinitionParser(OUTBOUND_ENDPOINT_MP, buildOutboundEndpointDefinitionParser());
        declareBeanDefinitionParser(SPY_MP, buildSpyDefinitionParser());
        declareBeanDefinitionParser(THROW_AN_MP, buildThrowAnDefinitionParser());
        declareBeanDefinitionParser(VERIFY_CALL_MP, buildVerifyCallDefinitionParser());
        declareBeanDefinitionParser(WHEN_MP, buildWhenDefinitionParser());
    }

    private MunitGenericDefinitionParser buildMockModuleDefinitionParser() {
        MunitGenericDefinitionParserBuilder builder = new MunitGenericDefinitionParserBuilder(MockModule.class);

        builder = builder.withScope("singleton")
                .withNoAttachProcessorDefinition();
        return builder.build();
    }

    private MunitGenericDefinitionParser buildOutboundEndpointDefinitionParser() {
        MunitGenericDefinitionParserBuilder builder = new MunitGenericDefinitionParserBuilder(OutboundEndpointMessageProcessor.class);

        builder = builder.withScope("prototype")
                .withAttribute(CONFIG_REF_ATT_NAME, "moduleObject")
                .withAttribute("address", "address")
                .withAttribute("returnPayload", "returnPayload")
                .withAttribute("exception-ref", "exception")
                .withAttribute("thenApplyTransformer-ref", "thenApplyTransformer")
                .withChildElement("return-invocation-properties", buildMapChildElement("return-invocation-properties", "returnInvocationProperties", "return-invocation-property"))
                .withChildElement("return-inbound-properties", buildMapChildElement("return-inbound-properties", "returnInboundProperties", "return-inbound-property"))
                .withChildElement("return-session-properties", buildMapChildElement("return-session-properties", "returnSessionProperties", "return-session-property"))
                .withChildElement("return-outbound-properties", buildMapChildElement("return-outbound-properties", "returnOutboundProperties", "return-outbound-property"))
                .withChildElement("assertions", buildNestedChildElement("assertions", "assertions"));

        return builder.build();
    }

    private MunitGenericDefinitionParser buildSpyDefinitionParser() {
        MunitGenericDefinitionParserBuilder builder = new MunitGenericDefinitionParserBuilder(SpyMessageProcessor.class);

        builder = builder.withScope("prototype")
                .withAttribute(CONFIG_REF_ATT_NAME, "moduleObject")
                .withAttribute("messageProcessor", "messageProcessor")
                .withChildElement(WITH_ATTRIBUTES_ATT_NAME, buildWithAttributesListlement())
                .withChildElement("assertions-before-call", buildNestedChildElement("assertions-before-call", "assertionsBeforeCall"))
                .withChildElement("assertions-after-call", buildNestedChildElement("assertions-after-call", "assertionsAfterCall"));

        return builder.build();
    }

    private MunitGenericDefinitionParser buildThrowAnDefinitionParser() {
        MunitGenericDefinitionParserBuilder builder = new MunitGenericDefinitionParserBuilder(ThrowAnMessageProcessor.class);

        builder = builder.withScope("prototype")
                .withAttribute(CONFIG_REF_ATT_NAME, "moduleObject")
                .withAttribute("exception-ref", "exception")
                .withAttribute("whenCalling", "whenCalling")
                .withChildElement(WITH_ATTRIBUTES_ATT_NAME, buildWithAttributesListlement());

        return builder.build();
    }

    private MunitGenericDefinitionParser buildVerifyCallDefinitionParser() {
        MunitGenericDefinitionParserBuilder builder = new MunitGenericDefinitionParserBuilder(VerifyCallMessageProcessor.class);

        builder = builder.withScope("prototype")
                .withAttribute(CONFIG_REF_ATT_NAME, "moduleObject")
                .withAttribute("messageProcessor", "messageProcessor")
                .withAttribute("times", "times")
                .withAttribute("atLeast", "atLeast")
                .withAttribute("atMost", "atMost")
                .withChildElement(WITH_ATTRIBUTES_ATT_NAME, buildWithAttributesListlement());

        return builder.build();
    }

    private MunitGenericDefinitionParser buildWhenDefinitionParser() {
        MunitGenericDefinitionParserBuilder builder = new MunitGenericDefinitionParserBuilder(WhenMessageProcessor.class);

        builder = builder.withScope("prototype")
                .withAttribute(CONFIG_REF_ATT_NAME, "moduleObject")
                .withAttribute("messageProcessor", "messageProcessor")
                .withChildElement(WITH_ATTRIBUTES_ATT_NAME, buildWithAttributesListlement())
                .withAttribute("thenApplyTransformer-ref", "thenApplyTransformer")
                .withChildElement("then-return", buildThenReturnChildElement());

        return builder.build();
    }

    private ParseableElement buildWithAttributesListlement() {
        ListChildElement.ListChildElementBuilder builder = new ListChildElement.ListChildElementBuilder(WITH_ATTRIBUTES_ATT_NAME, "withAttributes");
        builder.withItemClass(Attribute.class)
                .withItemName("with-attribute")
                .withItemAttribute("name", "name")
                .withItemAttribute("whereValue", "whereValue");

        return builder.build();
    }

    private ParseableElement buildThenReturnChildElement() {
        ChildElement.ChildElementBuilder builder = new ChildElement.ChildElementBuilder(MunitMuleMessage.class);

        builder.withElementName("then-return")
                .withElementPropertyName("thenReturn")
                .withAttribute("payload", "payload")
                .withChildElement(buildMapChildElement("invocation-properties", "invocationProperties", "invocation-property"))
                .withChildElement(buildMapChildElement("inbound-properties", "inboundProperties", "inbound-property"))
                .withChildElement(buildMapChildElement("session-properties", "sessionProperties", "session-property"))
                .withChildElement(buildMapChildElement("outbound-properties", "outboundProperties", "outbound-property"));

        return builder.build();
    }

    private ParseableElement buildMapChildElement(String elementName, String elementPropertyName, String elementItemName) {
        MapChildElement.MapChildElementBuilder builder = new MapChildElement.MapChildElementBuilder(elementName, elementPropertyName);
        builder.withItemName(elementItemName);
        return builder.build();
    }

    private NestedChildElement buildNestedChildElement(String elementName, String elementPropertyName) {
        NestedChildElement.NestedChildElementBuilder builder = new NestedChildElement.NestedChildElementBuilder();
        builder.withElementName(elementName)
                .withElementPropertyName(elementPropertyName);
        return builder.build();
    }
}
