/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.processors;

import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.transformer.TransformerMessagingException;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.mock.MockModule;
import org.mule.munit.mock.model.MunitMuleMessage;
import org.mule.munit.common.model.Property;
import org.mule.munit.wrapped.transformers.ExpressionEvaluatorTransformer;

import java.util.ArrayList;
import java.util.List;

/**
 * WhenMessageProcessor invokes the {@link org.mule.munit.mock.MockModule#when(String, java.util.List, org.mule.munit.mock.model.MunitMuleMessage, Object)} method in {@link org.mule.munit.mock.MockModule }. For each argument there is a field in this processor to match it.  Before invoking the actual method the processor will evaluate and transform where possible to the expected argument type.
 */
public class WhenMessageProcessor extends AbstractMockMessageProcessor {

    public static final String MESSAGE_PROCESSOR_NAME = "when";

    protected Object messageProcessor;
    protected Object withAttributes;
    protected Object thenReturn;
    protected Object thenApplyTransformer;

    /**
     * Sets withAttributes
     *
     * @param value Value to set
     */
    public void setWithAttributes(Object value) {
        this.withAttributes = value;
    }

    /**
     * Sets thenApplyTransformer
     *
     * @param value Value to set
     */
    public void setThenApplyTransformer(Object value) {
        this.thenApplyTransformer = value;
    }

    /**
     * Sets thenReturn
     *
     * @param value Value to set
     */
    public void setThenReturn(Object value) {
        this.thenReturn = value;
    }

    /**
     * Sets messageProcessor
     *
     * @param value Value to set
     */
    public void setMessageProcessor(Object value) {
        this.messageProcessor = value;
    }

    /**
     * Invokes the MessageProcessor.
     * <p/>
     *
     * @param muleEvent MuleEvent to be processed
     * @param module    the MockModule
     * @throws Exception
     */
    protected void doProcess(MuleEvent muleEvent, MockModule module) throws Exception {
        module.when(
                (String) messageProcessor,
                transformAttributes(withAttributes, muleEvent),
                transformThenReturn(muleEvent),
                evaluate(muleEvent.getMessage(), thenApplyTransformer)
        );
    }

    private MunitMuleMessage transformThenReturn(MuleEvent muleEvent) throws TransformerMessagingException {
        MunitMuleMessage munitMuleMessage = new MunitMuleMessage();

        if (thenReturn != null) {
            MunitMuleMessage originalThenReturn = (MunitMuleMessage) thenReturn;

            if (isInvalidExpression(originalThenReturn.getPayload())) {
                throw new MunitError("mock:then-return MEL parsing fail. The payload: " + originalThenReturn.getPayload() + "is an invalid expression.");
            }

            ExpressionEvaluatorTransformer evaluator = new ExpressionEvaluatorTransformer();
            munitMuleMessage.setPayload(evaluator.evaluate(this.muleContext, muleEvent, originalThenReturn.getPayload()));
            munitMuleMessage.setMimeType(((MunitMuleMessage) thenReturn).getMimeType());
            munitMuleMessage.setEncoding(((MunitMuleMessage) thenReturn).getEncoding());

            munitMuleMessage.setInvocationProperties(transformProperties(originalThenReturn.getInvocationProperties(), muleEvent, muleContext));
            munitMuleMessage.setInboundProperties(transformProperties(originalThenReturn.getInboundProperties(), muleEvent, muleContext));
            munitMuleMessage.setOutboundProperties(transformProperties(originalThenReturn.getOutboundProperties(), muleEvent, muleContext));
            munitMuleMessage.setSessionProperties(transformProperties(originalThenReturn.getSessionProperties(), muleEvent, muleContext));
        }
        return munitMuleMessage;
    }


    protected List<Property> transformProperties(Object properties, MuleEvent muleEvent, MuleContext muleContext) throws TransformerMessagingException {
        ExpressionEvaluatorTransformer evaluator = new ExpressionEvaluatorTransformer();

        List<Property> transformedProperties = new ArrayList<Property>();
        if (properties != null) {
            for (Property p : (List<Property>) properties) {
                Property result = new Property();
                if (isInvalidExpression(p.getKey())) {
                    throw new MunitError("mock:the-return MEL parsing fail. The key " + p.getKey() + "is an invalid expression.");
                }

                if (isInvalidExpression(p.getValue())) {
                    throw new MunitError("mock:the-return MEL parsing fail. The whereValue " + p.getValue() + "is an invalid expression.");
                }

                result.setKey((String) evaluator.evaluate(muleContext, muleEvent, p.getKey()));
                result.setValue(evaluator.evaluate(muleContext, muleEvent, p.getValue()));
                result.setMimeType(p.getMimeType());
                result.setEncoding(p.getEncoding());

                transformedProperties.add(result);
            }
        }

        return transformedProperties;
    }

    @Override
    protected String getProcessor() {
        return MESSAGE_PROCESSOR_NAME;
    }

}
