/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.behavior;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.common.behavior.BehaviorManager;
import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.behavior.ProcessorId;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Manages the mocking process.
 *
 * Gets the behaviors, stores the processor calls and stores the spy process
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class DefaultBehaviorManager implements BehaviorManager {

  public static String ID = "_munitBehaviorManager";

  private transient Log logger = LogFactory.getLog(this.getClass());

  /**
   * These are the real calls of the processors.
   */
  protected List<ProcessorCall> calls = new CopyOnWriteArrayList<>();

  /**
   * These are the behaviors expected for different processor mocks
   */
  protected List<MockBehavior> behaviors = new ArrayList<>();

  /**
   * The spy actions that are ran before a processor call
   */
  protected List<SpyBehavior> beforeCallProcessorSpies = new CopyOnWriteArrayList<>();

  /**
   * The spy actions that are ran after a processor call
   */
  protected List<SpyBehavior> afterCallProcessorSpies = new CopyOnWriteArrayList<>();

  @Override
  public void reset() {
    calls.clear();
    behaviors.clear();
    beforeCallProcessorSpies.clear();
    afterCallProcessorSpies.clear();
  }

  public synchronized void addBehavior(MockBehavior behavior) {
    behaviors.add(behavior);
  }

  public synchronized void addCall(ProcessorCall call) {
    calls.add(call);
  }

  public synchronized void addBeforeCallSpyAssertion(SpyBehavior spyBehavior) {
    beforeCallProcessorSpies.add(spyBehavior);
  }

  public synchronized void addAfterCallSpyAssertion(SpyBehavior spyBehavior) {
    afterCallProcessorSpies.add(spyBehavior);
  }

  /**
   * Gets the best matching Behavior. The best matching behavior is the one that mostly matches the attributes
   *
   * @param processorCall The comparing call
   * @return The best matching behavior
   */
  public MockBehavior getBetterMatchingBehavior(ProcessorCall processorCall) {
    return getBetterMatchingAction(processorCall, behaviors);
  }

  /**
   * Retrieve all the execute calls for a message processor that satisfies the attribute matchers
   *
   * @param mpId The Message processor Id
   * @param attributesMatchers The attributes that the message processor must match
   * @return The List of message processor calls
   */
  public List<ProcessorCall> findCallsFor(ProcessorId mpId, Map<String, Object> attributesMatchers) {
    logger.debug("Finding calls for message processor: " + mpId.getFullName());

    List<ProcessorCall> expected = new ArrayList<>();
    ProcessorCall matchingCall = new ProcessorCall(mpId);
    matchingCall.setAttributes(attributesMatchers);
    for (ProcessorCall call : calls) {
      if (matchingCall.matchingWeight(call) >= 0) {
        expected.add(call);
      }
    }
    return expected;
  }

  /**
   * Gets the best matching Before Spy assertion.
   *
   * @param processorCall The comparing call
   * @return The best matching Before spy assertion
   */
  public SpyBehavior getBetterMatchingBeforeSpyAssertion(ProcessorCall processorCall) {
    logger.debug("Finding Spy Before for: " + processorCall.getProcessorId().getFullName());
    return getBetterMatchingAction(processorCall, beforeCallProcessorSpies);
  }

  /**
   * Gets the best matching After Spy assertion.
   *
   * @param processorCall The comparing call
   * @return The best matching After spy assertion
   */
  public SpyBehavior getBetterMatchingAfterSpyAssertion(ProcessorCall processorCall) {
    logger.debug("Finding Spy After for: " + processorCall.getProcessorId().getFullName());
    return getBetterMatchingAction(processorCall, afterCallProcessorSpies);
  }

  protected <T extends Behavior> T getBetterMatchingAction(ProcessorCall processorCall, Collection<T> actions) {
    Map.Entry<Integer, T> bestMatchingBehavior = new AbstractMap.SimpleEntry<>(0, null);
    for (T behavior : actions) {
      int matchingWeight = behavior.getProcessorCall().matchingWeight(processorCall);
      if (matchingWeight >= 0 && matchingWeight >= bestMatchingBehavior.getKey()) {
        bestMatchingBehavior.setValue(behavior);
      }
    }

    return bestMatchingBehavior.getValue();
  }

  protected List<ProcessorCall> getCalls() {
    return new LinkedList<>(calls);
  }
}
