/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import java.nio.charset.Charset;
import java.util.Objects;
import java.util.Optional;

import org.mule.munit.common.event.DataTypeBuilder;
import org.mule.munit.common.event.MessageBuilderDecorator;
import org.mule.munit.common.model.Event;
import org.mule.munit.common.model.Payload;
import org.mule.munit.common.util.IOUtils;
import org.mule.runtime.api.interception.InterceptionEvent;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.MediaType;

/**
 * The objective of this class is to mutate the received InterceptionEvent.
 *
 * The nature of it is such that it can be mutated and there is no need to generate a new one.
 */
public class InterceptingEventBuilder {

  public InterceptionEvent build(InterceptionEvent originalEvent, Optional<Event> mockedEvent) {
    if (!mockedEvent.isPresent()) {
      return originalEvent;
    }

    if (null != mockedEvent.get().getVariables()) {
      mockedEvent.get().getVariables()
          .forEach(v -> originalEvent.addVariable(v.getKey(), v.getValue(), DataTypeBuilder.fromTypedKeyValue(v).build()));
    }
    // TODO MU-1073 Session properties are missing
    originalEvent.message(buildMessage(mockedEvent.get()));

    return originalEvent;
  }

  private Message buildMessage(Event mockedEvent) {
    MessageBuilderDecorator builder = new MessageBuilderDecorator();

    Payload mockedEventPayload = mockedEvent.getPayload();
    builder.withPayload(mockedEventPayload.getValue());
    if (mockedEventPayload.getMediaType() != null) {
      MediaType mediaType = MediaType.parse(mockedEventPayload.getMediaType());
      if (mockedEventPayload.getEncoding() != null) {
        mediaType = mediaType.withCharset(Charset.forName(mockedEventPayload.getEncoding()));
      }
      builder.withMediaType(mediaType);
    }

    if (null != mockedEvent.getInboundProperties()) {
      mockedEvent.getInboundProperties()
          .forEach(p -> builder.withInboundProperty(p.getKey(), p.getValue(), p.getMediaType(), p.getEncoding()));
    }

    if (null != mockedEvent.getOutboundProperties()) {
      mockedEvent.getOutboundProperties()
          .forEach(p -> builder.withOutboundProperty(p.getKey(), p.getValue(), p.getMediaType(), p.getEncoding()));
    }

    if (null != mockedEvent.getInboundAttachments()) {
      mockedEvent.getInboundAttachments().forEach(a -> builder.withInboundAttachment(a.getKey(), a.getDataHandler()));
    }

    if (null != mockedEvent.getOutboundAttachments()) {
      mockedEvent.getOutboundAttachments().forEach(a -> builder.withOutboundAttachment(a.getKey(), a.getDataHandler()));
    }

    return builder.build();
  }

}
