/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.mule.munit.common.behavior.BehaviorManager;
import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.behavior.ProcessorId;
import org.mule.munit.common.model.EventError;
import org.mule.munit.common.util.MunitUtils;
import org.mule.munit.mock.behavior.DefaultBehaviorManager;
import org.mule.munit.mock.behavior.MockBehavior;
import org.mule.munit.mock.behavior.SpyBehavior;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.interception.InterceptionAction;
import org.mule.runtime.api.interception.InterceptionEvent;
import org.mule.runtime.api.interception.ProcessorInterceptor;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.core.api.Event;
import org.mule.runtime.core.exception.ErrorTypeRepository;

/**
 * It's the actual MUnit Interceptor called before, during and after the execution of each processor.
 */
public class MunitProcessorInterceptor implements ProcessorInterceptor {

  private transient Log logger = LogFactory.getLog(this.getClass());

  private Map<String, Object> beforeParameters;
  private DefaultBehaviorManager manager;

  private ErrorTypeRepository errorTypeRepository;

  public void setManager(BehaviorManager manager) {
    this.manager = (DefaultBehaviorManager) manager;
  }

  protected DefaultBehaviorManager getManager() {
    if (manager == null) {
      throw new IllegalStateException("There is no manager defined");
    }
    return manager;
  }

  public void setErrorTypeRepository(ErrorTypeRepository errorTypeRepository) {
    this.errorTypeRepository = errorTypeRepository;
  }

  @Override
  public void before(ComponentLocation location, Map<String, Object> parameters, InterceptionEvent event) {
    logger.debug("About to run Spy Before processor: " + getIdentifier(location));

    setBeforeParameters(parameters);
    ProcessorCall processorCall = buildCall(getIdentifier(location), parameters);
    getManager().addCall(processorCall);

    // TODO We could make this work base on the interceptionEvent in any case it does not have sense without it
    // Event realEvent = null; // = buildEvent();
    // runSpyAssertion(getManager().getBetterMatchingBeforeSpyAssertion(messageProcessorCall), realEvent);
  }

  @Override
  public CompletableFuture<InterceptionEvent> around(ComponentLocation location, Map<String, Object> parameters,
                                                     InterceptionEvent event,
                                                     InterceptionAction action) {
    logger.debug("Retrieving mocked behavior for processor: " + getIdentifier(location));

    MockBehavior behavior = getManager().getBetterMatchingBehavior(buildCall(getIdentifier(location), parameters));
    if (behavior != null) {
      if (shouldFailProcessor(behavior)) {
        logger.debug("Mock behavior found. Throwing exception instead of " + getIdentifier(location));
        return failProcessor(event, action, behavior);
      }
      logger.debug("Mock behavior found. Executing that instead of " + getIdentifier(location));
      return returnBehavior(event, action, behavior);
    }

    return action.proceed();
  }

  @Override
  public void after(ComponentLocation location, InterceptionEvent event, Optional<Throwable> thrown) {
    logger.debug("About to run Spy After for processor: " + getIdentifier(location));

    // TODO We could make this work base on the interceptionEvent in any case it does not have sense without it
    // Event realEvent = null; // = buildEvent();
    // MunitMessageProcessorCall messageProcessorCall = buildCall(getIdentifier(), event, getParameters());
    // runSpyAssertion(getManager().getBetterMatchingBeforeSpyAssertion(messageProcessorCall), realEvent);
  }

  private ComponentIdentifier getIdentifier(ComponentLocation location) {
    return location.getComponentIdentifier().getIdentifier();
  }

  private synchronized Map<String, Object> getBeforeParameters() {
    return beforeParameters;
  }

  private synchronized void setBeforeParameters(Map<String, Object> beforeParameters) {
    this.beforeParameters = beforeParameters;
  }

  private ProcessorCall buildCall(ComponentIdentifier componentIdentifier, Map<String, Object> parameters) {
    ProcessorId id = new ProcessorId(componentIdentifier.getName(), componentIdentifier.getNamespace());
    ProcessorCall call = new ProcessorCall(id);
    call.setAttributes(parameters);
    return call;
  }

  private boolean shouldFailProcessor(MockBehavior behavior) {
    Optional<org.mule.munit.common.model.Event> behaviorEvent = behavior.getEvent();
    EventError behaviorError = behaviorEvent.isPresent() ? behaviorEvent.get().getError() : null;

    boolean isThereAnEvent = behaviorEvent.isPresent();
    boolean isThereAnErrorCause = behaviorError != null ? (Throwable) behaviorError.getCause() != null : false;
    boolean isThereAnErrorTypeId = behaviorError != null ? StringUtils.isNotBlank(behaviorError.getTypeId()) : false;

    return isThereAnEvent && (isThereAnErrorCause || isThereAnErrorTypeId);
  }

  private CompletableFuture<InterceptionEvent> returnBehavior(InterceptionEvent event, InterceptionAction action,
                                                              MockBehavior behavior) {
    action.skip();
    Optional<org.mule.munit.common.model.Event> mockedEvent = behavior.getEvent();
    return CompletableFuture.supplyAsync(() -> buildInterceptingEvent(event, mockedEvent));
  }

  private CompletableFuture<InterceptionEvent> failProcessor(InterceptionEvent event, InterceptionAction action,
                                                             MockBehavior behavior) {
    buildInterceptingEvent(event, behavior.getEvent());
    org.mule.munit.common.model.Event behaviorEvent = behavior.getEvent().get();
    if (behaviorEvent.getError().getCause() != null) {
      return action.fail((Throwable) behaviorEvent.getError().getCause());
    } else {
      ComponentIdentifier componentIdentifier =
          ComponentIdentifier.buildFromStringRepresentation(behaviorEvent.getError().getTypeId());
      Optional<ErrorType> errorType = errorTypeRepository.getErrorType(componentIdentifier);

      return action.fail(errorType.get());
    }
  }

  private InterceptionEvent buildInterceptingEvent(InterceptionEvent originalEvent,
                                                   Optional<org.mule.munit.common.model.Event> mockedEvent) {
    return new InterceptingEventBuilder().build(originalEvent, mockedEvent);
  }

  private void runSpyAssertion(SpyBehavior spyBehavior, Event event) {
    if (spyBehavior == null) {
      logger.debug("No Spy was found to be run.");
      return;
    }

    logger.debug("Running Spy");
    MunitUtils.verifyAssertions(event, spyBehavior.getProcessors());
  }

}
