/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.PROCESSOR;

import java.util.Arrays;
import java.util.List;

import org.mule.munit.common.behavior.BehaviorManager;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.interception.ProcessorInterceptor;
import org.mule.runtime.api.interception.ProcessorInterceptorFactory;
import org.mule.runtime.core.exception.ErrorTypeRepository;

/**
 * The goal of this class is to define if a processor should be intercepted. If so create and store temporarily a
 * ProcessorInterceptor
 */
public class MunitProcessorInterceptorFactory implements ProcessorInterceptorFactory {

  private static final List<String> NON_INTERCEPTABLE_NAMESPACES = Arrays.asList("munit", "munit-tools", "mock");

  private BehaviorManager behaviorManager;
  private ErrorTypeRepository errorTypeRepository;

  public void setBehaviorManager(BehaviorManager behaviorManager) {
    this.behaviorManager = behaviorManager;
  }

  public void setErrorTypeRepository(ErrorTypeRepository errorTypeRepository) {
    this.errorTypeRepository = errorTypeRepository;
  }

  @Override
  public ProcessorInterceptor get() {
    return createInterceptor(behaviorManager, errorTypeRepository);
  }

  @Override
  public boolean intercept(ComponentLocation location) {
    TypedComponentIdentifier identifier = location.getComponentIdentifier();
    if (NON_INTERCEPTABLE_NAMESPACES.contains(identifier.getIdentifier().getNamespace())) {
      return false;
    }
    if (!PROCESSOR.equals(identifier.getType()) && !OPERATION.equals(identifier.getType())) {
      return false;
    }

    return true;
  }

  protected MunitProcessorInterceptor createInterceptor(BehaviorManager behaviorManager,
                                                        ErrorTypeRepository errorTypeRepository) {

    MunitProcessorInterceptor interceptor = new MunitProcessorInterceptor();
    interceptor.setManager(behaviorManager);
    interceptor.setErrorTypeRepository(errorTypeRepository);
    return interceptor;
  }

}
