/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.processors;

import org.apache.commons.lang3.StringUtils;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.model.Attribute;
import org.mule.munit.common.util.MunitExpressionWrapper;
import org.mule.munit.mock.MockModule;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.meta.AbstractAnnotatedObject;
import org.mule.runtime.core.api.Event;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.NestedProcessor;
import org.mule.runtime.core.api.config.ConfigurationException;
import org.mule.runtime.core.api.construct.FlowConstruct;
import org.mule.runtime.core.api.construct.FlowConstructAware;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.core.api.registry.RegistrationException;
import org.mule.runtime.core.config.i18n.CoreMessages;
import org.mule.runtime.core.exception.MessagingException;
import org.mule.runtime.core.processor.chain.NestedProcessorChain;

import java.util.ArrayList;
import java.util.List;


/**
 *
 * Generic Mock Processor.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
// TODO this may not be needed anymore
public abstract class AbstractMockProcessor extends AbstractAnnotatedObject
    implements Initialisable, Processor, FlowConstructAware {

  protected Object moduleObject;

  protected MuleContext muleContext;
  protected FlowConstruct flowConstruct;
  protected MunitExpressionWrapper expressionWrapper;

  protected String processor;

  protected List<Attribute> withAttributes;

  public String getProcessor() {
    return processor;
  }

  public void setProcessor(String processor) {
    this.processor = processor;
  }

  public List<Attribute> getWithAttributes() {
    return withAttributes;
  }

  public void setWithAttributes(List<Attribute> withAttributes) {
    this.withAttributes = withAttributes;
  }

  public void initialise() throws InitialisationException {
    expressionWrapper = new MunitExpressionWrapper(muleContext.getExpressionManager());

    try {
      moduleObject = findOrCreate(MockModule.class, true);
    } catch (IllegalAccessException | InstantiationException | ConfigurationException e) {
      throw new InitialisationException(CoreMessages.createStaticMessage("Cannot find object"), this);
    } catch (RegistrationException e) {
      throw new InitialisationException(CoreMessages.initialisationFailure(MockModule.class.getCanonicalName()), e, this);
    }
  }

  protected final Object findOrCreate(Class<?> moduleClass, boolean shouldAutoCreate)
      throws IllegalAccessException, InstantiationException, ConfigurationException, RegistrationException {

    Object temporaryObject = moduleObject;

    if (temporaryObject == null) {
      temporaryObject = (muleContext.getRegistry().lookupObject(moduleClass));
      if (temporaryObject == null) {
        if (shouldAutoCreate) {
          temporaryObject = (moduleClass.newInstance());
          muleContext.getRegistry().registerObject(moduleClass.getName(), temporaryObject);
        } else {
          throw new ConfigurationException(CoreMessages.createStaticMessage("Cannot find object"));
        }
      }
    }
    return temporaryObject;
  }

  protected MockModule getModule(Event event, String methodName) throws MessagingException {
    MockModule castedModuleObject;
    if (moduleObject instanceof String) {
      castedModuleObject = (muleContext.getRegistry().lookupObject(((String) moduleObject)));
      if (castedModuleObject == null) {
        throw new MessagingException(CoreMessages.failedToCreate(methodName), event, new RuntimeException(
                                                                                                          "Cannot find the configuration specified by the org.mule.munit.config-ref attribute."));
      }
    } else {
      castedModuleObject = ((MockModule) moduleObject);
    }
    return castedModuleObject;
  }


  /**
   * Executes the message processor code. In case of an assertion error it throws a new exception with the mule stack trace
   *
   * @param event The mule event to be processed.
   * @return The result mule event
   * @throws MuleException In case of error. If the assertion fails, it throws an {@link AssertionError}
   * @since 1.0.0
   */
  public Event process(Event event) throws MuleException {
    MockModule module = getModule(event, getProcessorName());
    try {
      doProcess(event, module);
      return event;
    } catch (AssertionError error) {
      AssertionError exception = new AssertionError(getMessage(error));
      throw exception;
    } catch (Exception e) {
      throw new MessagingException(CoreMessages.failedToInvoke(getProcessorName()), event, e);
    }
  }


  /**
   * The method that do the actual process
   *
   * @param muleEvent The mule Message
   * @param module The instance of the mock module
   */
  protected abstract void doProcess(Event muleEvent, MockModule module) throws Exception;

  /**
   * @return The name of the processor
   */
  protected abstract String getProcessorName();

  public void setMuleContext(MuleContext context) {
    this.muleContext = context;
  }

  public void setModuleObject(Object moduleObject) {
    this.moduleObject = moduleObject;
  }

  public void setFlowConstruct(FlowConstruct flowConstruct) {
    this.flowConstruct = flowConstruct;
  }

  private String getMessage(AssertionError error) {
    String message = error.getMessage();
    if (StringUtils.isEmpty(message)) {
      return this.getProcessorName();
    }
    return message;
  }

  protected List<Attribute> getEvaluatedAttributes(Object attributes, Event event) {
    List<Attribute> evaluatedAttributes = new ArrayList<>();
    if (attributes != null) {
      for (Attribute attribute : (List<Attribute>) attributes) {
        if (isInvalidExpression(attribute.getAttributeName())) {
          throw new MunitError(
                               "mock:with-attributes parsing fail. The name " + attribute.getAttributeName()
                                   + "is an invalid expression.");
        }

        if (isInvalidExpression(attribute.getWhereValue())) {
          throw new MunitError("mock:with-attributes parsing fail. The whereValue " + attribute.getWhereValue()
              + "is an invalid expression.");
        }

        evaluatedAttributes.add(attribute.getEvaluatedAttribute(event, expressionWrapper));
      }
    }

    return evaluatedAttributes;
  }

  protected boolean isInvalidExpression(Object expression) {
    return expressionWrapper.isExpression(expression) && !expressionWrapper.isExpressionValid(expression);
  }

  protected List<NestedProcessor> transformAssertions(Object assertions, Event event) {
    final List<NestedProcessor> transformedAssertions = new ArrayList<>();
    if (assertions != null) {
      for (Processor processor : ((List<Processor>) assertions)) {
        transformedAssertions.add(new NestedProcessorChain(event, processor));
      }
    }
    return transformedAssertions;
  }

}
