/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.tool;

import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.behavior.ProcessorId;
import org.mule.munit.common.model.Event;
import org.mule.munit.mock.behavior.DefaultBehaviorManager;
import org.mule.munit.mock.behavior.MockBehavior;
import org.mule.runtime.core.api.MuleContext;

import java.util.Map;

/**
 * Mocking Tool to create processor mocks.
 *
 * Usage: new ProcessorMocker(muleContext).when("processors").ofNamespace("namespace").thenReturn(muleMessage);
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class ProcessorMocker extends AbstractMockingTool {

  public ProcessorMocker(MuleContext muleContext) {
    super(muleContext);
  }

  /**
   * Defines the name of the processor to be mocked
   *
   * @param name The name of the processor to be mocked
   * @return this
   */
  public ProcessorMocker when(String name) {
    this.processorName = name;
    return this;
  }

  /**
   * Defines the namespace of the processor to be mocked
   *
   * @param namespace The namespace of the processor to be mocked
   * 
   * @return this
   *
   */
  public ProcessorMocker ofNamespace(String namespace) {
    this.processorNamespace = namespace;
    return this;
  }

  /**
   * Defines the attributes of the processor to be mocked
   *
   * @param attributes The attributes of the processor to be mocked
   * @return this
   */
  public ProcessorMocker withAttributes(Map<String, Object> attributes) {
    this.processorAttributes = attributes;
    return this;
  }

  /**
   * Defines what {@link Event} to return after the processor call
   *
   * @param event The Event to return
   */
  // TODO change this to return events MU-641
  public void thenReturn(Event event) {
    validateMessageProcessorName();

    ProcessorCall processorCall = new ProcessorCall(new ProcessorId(processorName, processorNamespace));
    processorCall.setAttributes(processorAttributes);
    getManager().addBehavior(new MockBehavior(processorCall, event));
  }

  /**
   * Determines that the mocked processor must return the same event as before its call.
   */
  public void thenReturnSameEvent() {
    validateMessageProcessorName();

    ProcessorCall processorCall = new ProcessorCall(new ProcessorId(processorName, processorNamespace));
    processorCall.setAttributes(processorAttributes);

    DefaultBehaviorManager manager = getManager();
    manager.addBehavior(new MockBehavior(processorCall));
  }

  private void validateMessageProcessorName() {
    if (processorName == null) {
      throw new IllegalArgumentException("You must specify at least the processor name");
    }
  }
}
