/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Arrays.asList;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ERROR_HANDLER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.FLOW;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.INTERCEPTING;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ON_ERROR;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SCOPE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SOURCE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;

import java.util.List;

import org.mule.munit.common.model.Event;
import org.mule.munit.common.model.NullObject;
import org.mule.munit.mock.behavior.MockBehavior;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The class validates if a location should or should not be mocked. It also validates if a behavior definition is valid for an
 * specific location
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MockingValidator {

  public static final List<String> NON_MOCKABLE_CORE_OPERATIONS = asList("logger", "ee:transform");
  public static final List<String> SPECIAL_CORE_OPERATIONS = asList("set-payload", "set-variable");

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private ComponentLocation location;

  public MockingValidator(ComponentLocation location) {
    checkNotNull(location, "The location must not be null.");
    this.location = location;
  }

  /**
   * Validates if the {@link ComponentLocation} allow to be spy or not. It works a black list, allowing spy for any component
   * location save those black listed.
   *
   * @return false if the {@link ComponentLocation} does not allow mocking, true otherwise
   */
  public Boolean allowMocking() {
    ComponentType type = location.getComponentIdentifier().getType();
    if (type.equals(FLOW) || type.equals(SOURCE) || type.equals(SCOPE) || type.equals(ROUTER) || type.equals(INTERCEPTING)
        || type.equals(ERROR_HANDLER) || type.equals(ON_ERROR) || type.equals(UNKNOWN)) {
      return false;
    }

    if (type.equals(OPERATION)) {
      if (NON_MOCKABLE_CORE_OPERATIONS.contains(location.getComponentIdentifier().getIdentifier().toString())) {
        return false;
      }
    }

    return true;
  }

  /**
   * Based on the given {@link ComponentLocation} validates if the {@link MockBehavior} tries to set an invalid value.
   *
   * @param behavior a {@link MockBehavior}
   * @return false if the behavior attempts to set invalid values for the given location, true otherwise
   */
  public Boolean isBehaviorValid(MockBehavior behavior) {
    if (!allowMocking()) {
      logger.error(getBaseErrorMessage() + " This component does not allow mocking");
      return false;
    }

    ComponentType type = location.getComponentIdentifier().getType();
    switch (type) {
      case OPERATION:
        return isBehaviorValidForOperation(behavior);
      default:
        return true;
    }
  }

  private boolean isBehaviorValidForOperation(MockBehavior behavior) {
    if (!behavior.getEvent().isPresent()) {
      return true;
    }

    if ("flow-ref".equals(location.getComponentIdentifier().getIdentifier().toString())) {
      return true;
    }

    String baseErrorMessage = getBaseErrorMessage() + " This component does not allow definition of";

    if (SPECIAL_CORE_OPERATIONS.contains(location.getComponentIdentifier().getIdentifier().toString())) {
      return isBehaviorValidForCoreOperations(behavior, baseErrorMessage);

    } else {
      return isBehaviorValidForCommonNotAllowedElements(behavior, baseErrorMessage);
    }
  }

  private boolean isBehaviorValidForCoreOperations(MockBehavior behavior, String baseErrorMessage) {

    if ("set-payload".equals(location.getComponentIdentifier().getIdentifier().toString())) {
      return isBehaviorValidForSetPayload(behavior, baseErrorMessage);
    }
    if ("set-variable".equals(location.getComponentIdentifier().getIdentifier().toString())) {
      return isBehaviorValidForSetVariable(behavior, baseErrorMessage);
    }

    return true;
  }

  private boolean isBehaviorValidForSetPayload(MockBehavior behavior, String baseErrorMessage) {
    Event event = behavior.getEvent().get();
    if (!isBehaviorValidForCommonNotAllowedElements(behavior, baseErrorMessage)) {
      return false;
    }
    if (!(event.getAttributes() == null || event.getAttributes().getValue() instanceof NullObject)) {
      logger.error(baseErrorMessage + " attributes.");
      return false;
    }
    if (event.getVariables() != null) {
      logger.error(baseErrorMessage + " variables.");
      return false;
    }
    return true;
  }

  private boolean isBehaviorValidForSetVariable(MockBehavior behavior, String baseErrorMessage) {
    Event event = behavior.getEvent().get();
    if (!isBehaviorValidForCommonNotAllowedElements(behavior, baseErrorMessage)) {
      return false;
    }
    if (!(event.getAttributes() == null || event.getAttributes().getValue() instanceof NullObject)) {
      logger.error(baseErrorMessage + " attributes.");
      return false;
    }
    if (!(event.getPayload() == null || event.getPayload().getValue() instanceof NullObject)) {
      logger.error(baseErrorMessage + " payload.");
      return false;
    }

    return true;
  }

  private Boolean isBehaviorValidForCommonNotAllowedElements(MockBehavior behavior, String baseErrorMessage) {
    Event event = behavior.getEvent().get();
    if (event.getSessionProperties() != null) {
      logger.error(baseErrorMessage + " session properties.");
      return false;
    }
    if (event.getOutboundProperties() != null) {
      logger.error(baseErrorMessage + " outbound properties.");
      return false;
    }
    if (event.getInboundProperties() != null) {
      logger.error(baseErrorMessage + " inbound properties.");
      return false;
    }
    if (event.getInboundAttachments() != null) {
      logger.error(baseErrorMessage + " inbound attachments.");
      return false;
    }
    if (event.getOutboundAttachments() != null) {
      logger.error(baseErrorMessage + " outbound attachments.");
      return false;
    }
    return true;
  }

  public String getBaseErrorMessage() {
    String identifier = location.getComponentIdentifier().getIdentifier().toString();
    String baseErrorMessage = "You have defined a behavior for "
        + identifier + " in "
        + location.getFileName().orElse(" ? ")
        + "[line: " + location.getLineInFile().orElse(-1) + "].";
    return baseErrorMessage;
  }

}
