/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import org.mule.munit.common.model.Event;
import org.mule.munit.common.model.NullObject;
import org.mule.munit.mock.behavior.Behavior;
import org.mule.munit.mock.behavior.CallBehaviour;
import org.mule.munit.mock.behavior.MockBehavior;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Arrays.asList;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ERROR_HANDLER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.FLOW;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.INTERCEPTING;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ON_ERROR;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SCOPE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SOURCE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;

/**
 * Validate if a location should be mocked. It also checks if a behaviour definition is valid for a specific location.
 *
 * @author Mulesoft Inc.
 */
class BehaviourValidator {

  private static final String LOGGER = "logger";
  private static final String TRANSFORM = "ee:transform";
  private static final String FLOW_REF = "flow-ref";
  private static final String SET_PAYLOAD = "set-payload";
  private static final String SET_VARIABLE = "set-variable";

  private static final List<ComponentType> NON_MOCKABLE_TYPES =
      asList(FLOW, SOURCE, SCOPE, ROUTER, INTERCEPTING, ERROR_HANDLER, ON_ERROR, UNKNOWN);
  private static final List<String> NON_MOCKABLE_CORE_OPERATIONS = asList(LOGGER, TRANSFORM);
  private static final List<String> SPECIAL_CORE_OPERATIONS = asList(SET_PAYLOAD, SET_VARIABLE);

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private ComponentLocation location;

  BehaviourValidator(ComponentLocation location) {
    checkNotNull(location, "The location must not be null.");
    this.location = location;
  }

  /**
   * Validates if the {@link ComponentLocation} allows to be mocked or not. It works as a blacklist, allowing mock for any
   * component location save those here blacklisted.
   *
   * @return {@code false} if the {@link ComponentLocation} does not allow mocking, {@code true} otherwise
   */
  Boolean allowMocking() {
    ComponentType type = location.getComponentIdentifier().getType();
    if (NON_MOCKABLE_TYPES.contains(type)) {
      return false;
    } else {
      return !NON_MOCKABLE_CORE_OPERATIONS.contains(location.getComponentIdentifier().getIdentifier().toString());
    }
  }

  /**
   * Based on the given {@link ComponentLocation} validate if the {@link Behavior} tries to set an invalid value.
   *
   * @param behaviour a {@link Behavior} to check
   * @return {@code false} if the behaviour attempts to set invalid values for the given location, {@code true} otherwise
   */
  Boolean isBehaviorValid(Behavior behaviour) {
    if (!allowMocking()) {
      logger.error(getBaseErrorMessage() + " This component does not allow mocking");
      return false;
    }

    if (behaviour instanceof CallBehaviour && !behaviour.getEvent().isPresent()) {
      logger.error(getBaseErrorMessage() + " CallBehaviour returned no event");
      return false;
    }

    return !location.getComponentIdentifier().getType().equals(OPERATION) || isBehaviorValidForOperation(behaviour);
  }

  String getBaseErrorMessage() {
    String identifier = location.getComponentIdentifier().getIdentifier().toString();
    return String.format("You have defined a behaviour for %s in %s [line: %s].",
                         identifier, location.getFileName().orElse("?"), location.getLineInFile().orElse(-1));
  }

  private boolean isBehaviorValidForOperation(Behavior behaviour) {
    if (!behaviour.getEvent().isPresent()) {
      if (behaviour instanceof MockBehavior) { // Should return the same event that was received
        return true;
      } else if (behaviour instanceof CallBehaviour) { // CallBehaviour was not evaluated
        return false;
      }
    }

    if (FLOW_REF.equals(location.getComponentIdentifier().getIdentifier().toString())) {
      return true;
    }

    String baseErrorMessage = getBaseErrorMessage() + " This component does not allow definition of";

    if (SPECIAL_CORE_OPERATIONS.contains(location.getComponentIdentifier().getIdentifier().toString())) {
      return isBehaviorValidForCoreOperations(behaviour, baseErrorMessage);
    } else {
      return isBehaviorValidForCommonNotAllowedElements(behaviour, baseErrorMessage);
    }
  }

  private boolean isBehaviorValidForCoreOperations(Behavior behaviour, String baseErrorMessage) {
    String identifier = location.getComponentIdentifier().getIdentifier().toString();

    if (SET_PAYLOAD.equals(identifier)) {
      return isBehaviorValidForSetPayload(behaviour, baseErrorMessage);
    } else {
      return isBehaviorValidForSetVariable(behaviour, baseErrorMessage);
    }
  }

  // Event should always be present by this point
  @SuppressWarnings("OptionalGetWithoutIsPresent")
  private boolean isBehaviorValidForSetPayload(Behavior behaviour, String baseErrorMessage) {
    if (!isBehaviorValidForCommonNotAllowedElements(behaviour, baseErrorMessage)) {
      return false;
    }

    Event event = behaviour.getEvent().get();

    if (!(event.getAttributes() == null || event.getAttributes().getValue() instanceof NullObject)) {
      logger.error(baseErrorMessage + " attributes.");
      return false;
    }
    if (event.getVariables() != null) {
      logger.error(baseErrorMessage + " variables.");
      return false;
    }

    return true;
  }

  // Event should always be present by this point
  @SuppressWarnings("OptionalGetWithoutIsPresent")
  private boolean isBehaviorValidForSetVariable(Behavior behaviour, String baseErrorMessage) {
    if (!isBehaviorValidForCommonNotAllowedElements(behaviour, baseErrorMessage)) {
      return false;
    }

    Event event = behaviour.getEvent().get();

    if (!(event.getAttributes() == null || event.getAttributes().getValue() instanceof NullObject)) {
      logger.error(baseErrorMessage + " attributes.");
      return false;
    }

    if (!(event.getPayload() == null || event.getPayload().getValue() instanceof NullObject)) {
      logger.error(baseErrorMessage + " payload.");
      return false;
    }

    return true;
  }

  // Event should always be present by this point
  @SuppressWarnings("OptionalGetWithoutIsPresent")
  private Boolean isBehaviorValidForCommonNotAllowedElements(Behavior behaviour, String baseErrorMessage) {
    Event event = behaviour.getEvent().get();

    if (event.getSessionProperties() != null) {
      logger.error(baseErrorMessage + " session properties.");
      return false;
    }
    if (event.getOutboundProperties() != null) {
      logger.error(baseErrorMessage + " outbound properties.");
      return false;
    }
    if (event.getInboundProperties() != null) {
      logger.error(baseErrorMessage + " inbound properties.");
      return false;
    }
    if (event.getInboundAttachments() != null) {
      logger.error(baseErrorMessage + " inbound attachments.");
      return false;
    }
    if (event.getOutboundAttachments() != null) {
      logger.error(baseErrorMessage + " outbound attachments.");
      return false;
    }

    return true;
  }
}
