/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.behavior;

import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.model.EventAttributes;
import org.mule.munit.common.model.EventError;
import org.mule.munit.common.model.Payload;
import org.mule.munit.common.model.Property;
import org.mule.munit.common.model.Variable;
import org.mule.runtime.api.component.execution.ComponentExecutionException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.core.privileged.event.MuleSession;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;

/**
 * The representation of a processor mocked behavior that calls a flow to define which event should be returned.
 *
 * It is used as a replacement of the real processor. We use this in order to know that the processor must return.
 *
 * NOTE: MuleSession used in {@link #evaluate(MuleSession, org.mule.runtime.api.event.Event)} is deprecated, but the
 * {@link org.mule.munit.mock.interception.InterceptingEventBuilder} depends on it for session properties
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
@SuppressWarnings("deprecation")
public class CallBehaviour extends Behavior {

  private Flow flow;

  /**
   * Create a behaviour for calling a flow when mocking a processor.
   *
   * @param processorCall the processor call to be mocked
   * @param flow the {@link Flow} that will be used
   */
  public CallBehaviour(ProcessorCall processorCall, Flow flow) {
    super(processorCall);
    this.flow = flow;
  }

  /**
   * Invoke the associated flow and store the event, to avoid invoking the flow more than once. Following evaluations will have no
   * effect until {@link #clearEvent()} is invoked.
   *
   * @param session the service session for the event
   * @param input the event that was intercepted during mocking
   * @throws MunitError if {@link Flow#execute(org.mule.runtime.api.event.Event)} fails
   */
  public void evaluate(MuleSession session, org.mule.runtime.api.event.Event input) {
    if (!getEvent().isPresent()) {
      try {
        org.mule.runtime.api.event.Event output = flow.execute(input).get();
        setEvent(buildEvent(session, output));
      } catch (ComponentExecutionException cause) {
        setEvent(buildEvent(session, cause.getEvent()));
      } catch (Throwable cause) {
        if (cause.getCause() instanceof ComponentExecutionException) {
          setEvent(buildEvent(session, ((ComponentExecutionException) cause.getCause()).getEvent()));
        } else {
          throw new MunitError(String.format("There was a problem while evaluating '%s'", flow.getName()), cause);
        }
      }
    }
  }

  /**
   * Clear the event set in {@link #evaluate(MuleSession, org.mule.runtime.api.event.Event)}. Has no effect if the method was not
   * called, or if the event was already cleared.
   */
  public void clearEvent() {
    setEvent(null);
  }

  private org.mule.munit.common.model.Event buildEvent(MuleSession session, org.mule.runtime.api.event.Event event) {
    org.mule.munit.common.model.Event munitEvent = new org.mule.munit.common.model.Event();

    munitEvent.setPayload(buildPayload(event));
    munitEvent.setAttributes(buildAttributes(event));
    munitEvent.setError(buildError(event));
    munitEvent.setVariables(buildVariables(event));
    munitEvent.setSessionProperties(buildSessionProperties(session));

    return munitEvent;
  }

  private Payload buildPayload(org.mule.runtime.api.event.Event event) {
    TypedValue<?> corePayload = event.getMessage().getPayload();
    MediaType coreMediaType = corePayload.getDataType().getMediaType();

    Payload munitPayload = new Payload();
    munitPayload.setValue(corePayload.getValue());
    munitPayload.setMediaType(coreMediaType.withoutParameters().toString());
    coreMediaType.getCharset().ifPresent(charset -> munitPayload.setEncoding(charset.toString()));

    return munitPayload;
  }

  private EventAttributes buildAttributes(org.mule.runtime.api.event.Event event) {
    TypedValue<?> coreAttributes = event.getMessage().getAttributes();

    if (coreAttributes != null) {
      EventAttributes munitAttributes = new EventAttributes();
      MediaType coreMediaType = coreAttributes.getDataType().getMediaType();

      munitAttributes.setValue(coreAttributes.getValue());
      munitAttributes.setMediaType(coreMediaType.withoutParameters().toString());
      coreMediaType.getCharset().ifPresent(charset -> munitAttributes.setEncoding(charset.toString()));

      return munitAttributes;
    }

    return null;
  }

  private EventError buildError(org.mule.runtime.api.event.Event event) {
    EventError munitError = new EventError();
    if (event.getError().isPresent()) {
      munitError.setCause(event.getError().get().getCause());
      munitError.setTypeId(event.getError().get().getErrorType().getIdentifier());
    }

    return munitError;
  }

  private List<Variable> buildVariables(org.mule.runtime.api.event.Event event) {
    Map<String, TypedValue<?>> coreEventVariables = event.getVariables();
    List<Variable> munitVariables = new ArrayList<>();
    for (String key : coreEventVariables.keySet()) {
      Variable munitVariable = new Variable();
      munitVariable.setKey(key);
      munitVariable.setValue((Serializable) coreEventVariables.get(key).getValue());
      coreEventVariables.get(key).getDataType().getMediaType().getCharset()
          .ifPresent(charset -> munitVariable.setEncoding(charset.toString()));
      MediaType mimeType = coreEventVariables.get(key).getDataType().getMediaType();

      if (!mimeType.equals(MediaType.ANY)) {
        munitVariable.setMediaType(mimeType.withoutParameters().toString());
      }

      munitVariables.add(munitVariable);
    }

    return munitVariables;
  }

  private List<Property> buildSessionProperties(MuleSession session) {
    List<Property> munitProperties = new ArrayList<>();
    for (String key : session.getPropertyNamesAsSet()) {
      if (session.getProperty(key) instanceof Serializable) {
        Property munitProperty = new Property();

        munitProperty.setKey(key);
        munitProperty.setValue((Serializable) session.getProperty(key));
        session.getPropertyDataType(key).getMediaType().getCharset()
            .ifPresent(charset -> munitProperty.setEncoding(charset.toString()));
        munitProperty.setMediaType(session.getPropertyDataType(key).getMediaType().withoutParameters().toString());

        munitProperties.add(munitProperty);
      }
    }

    return munitProperties;
  }
}
